﻿---
title: Basic configuration
description: This page explains the basic configuration options for the JavaScript client. The Elasticsearch endpoint to use. It can be a single string or an array...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/javascript/basic-config
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch JavaScript Client
---

# Basic configuration
This page explains the basic configuration options for the JavaScript client.
```js
const { Client } = require('@elastic/elasticsearch')

const client = new Client({
  cloud: { id: '<cloud-id>' },
  auth: { apiKey: 'base64EncodedKey' },
  maxRetries: 5,
  sniffOnStart: true
})
```


### `node` or `nodes`

The Elasticsearch endpoint to use. It can be a single string or an array of strings:
```js
node: 'http://localhost:9200'
```

```js
nodes: ['http://localhost:9200', 'http://localhost:9201']
```

Or it can be an object (or an array of objects) that represents the node:
```js
node: {
  url: new URL('http://localhost:9200'),
  tls: 'tls options',
  agent: 'http agent options',
  id: 'custom node id',
  headers: { 'custom': 'headers' },
  roles: {
    master: true,
    data: true,
    ingest: true,
    ml: false
  }
}
```

---


### `auth`

Default: `null`
Your authentication data. You can use both basic authentication and [ApiKey](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-security-create-api-key).
See [Authentication](/docs/reference/elasticsearch/clients/javascript/connecting#authentication) for more details.
Basic authentication:
```js
auth: {
  username: 'elastic',
  password: 'changeme'
}
```

[ApiKey](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-security-create-api-key) authentication:
```js
auth: {
  apiKey: 'base64EncodedKey'
}
```

Bearer authentication, useful for [service account tokens](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-security-create-service-token). Be aware that it does not handle automatic token refresh:
```js
auth: {
  bearer: 'token'
}
```


### `maxRetries`

Type: `number`
Default: `3`
Max number of retries for each request.

### `requestTimeout`

Type: `number`
Default: `No value`
Max request timeout in milliseconds for each request.

### `pingTimeout`

Type: `number`
Default: `3000`
Max ping request timeout in milliseconds for each request.

### `sniffInterval`

Type: `number, boolean`
Default: `false`
Perform a sniff operation every `n` milliseconds.
<tip>
  Sniffing might not be the best solution. Before using the various `sniff` options, review this [blog post](https://www.elastic.co/blog/elasticsearch-sniffing-best-practices-what-when-why-how).
</tip>


### `sniffOnStart`

Type: `boolean`
Default: `false`
Perform a sniff once the client is started. Be sure to review the sniffing best practices [blog post](https://www.elastic.co/blog/elasticsearch-sniffing-best-practices-what-when-why-how).

### `sniffEndpoint`

Type: `string`
Default: `'_nodes/_all/http'`
Endpoint to ping during a sniff. Be sure to review the sniffing best practices [blog post](https://www.elastic.co/blog/elasticsearch-sniffing-best-practices-what-when-why-how).

### `sniffOnConnectionFault`

Type: `boolean`
Default: `false`
Perform a sniff on connection fault. Be sure to review the sniffing best practices [blog post](https://www.elastic.co/blog/elasticsearch-sniffing-best-practices-what-when-why-how).

### `resurrectStrategy`

Type: `string`
Default: `'ping'`
Configure the node resurrection strategy.
Options: `'ping'`, `'optimistic'`, `'none'`

### `suggestCompression`

Type: `boolean`
Default: `false`
Adds an `accept-encoding` header to every request.

### `compression`

Type: `string, boolean`
Default: `false`
Enables gzip request body compression.
Options: `'gzip'`, `false`

### `tls`

Type: `http.SecureContextOptions`
Default: `null`
The [tls configuraton](https://nodejs.org/api/tls.html).

### `proxy`

Type: `string, URL`
Default: `null`
If you are using an http(s) proxy, you can put its url here. The client will automatically handle the connection to it.
```js
const client = new Client({
  node: 'http://localhost:9200',
  proxy: 'http://localhost:8080'
})

const client = new Client({
  node: 'http://localhost:9200',
  proxy: 'http://user:pwd@localhost:8080'
})
```


### `agent`

Type: `http.AgentOptions, undici.PoolOptions, function, false`
Default: `null`
If using the default `UndiciConnection` from `@elastic/transport`, this value can be:
- an [Undici `PoolOptions` object](https://undici.nodejs.org/#/docs/api/Pool?id=parameter-pooloptions)
- a function that receives all connection-related options and returns an [Undici `Agent`](https://undici.nodejs.org/#/docs/api/Agent.md) instance (or any other object that follows [Undici's `Dispatch.request()`](https://undici.nodejs.org/#/docs/api/Dispatcher?id=dispatcherrequestoptions-callback) conventions)

If using the legacy `HttpConnection` from `@elastic/transport`, this value can be:
- [the options object passed to an `http.Agent`](https://nodejs.org/api/http.html#new-agentoptions)
- a function that returns an `http.Agent` (and thus also an [`https.Agent`](https://nodejs.org/api/https.html#class-httpsagent), or any implementation that follows the same conventions, like [`hpagent`](https://www.npmjs.com/package/hpagent))
- `false` to disable all agent usage, including the `keep-alive` feature

```js
const client = new Client({
  node: 'http://localhost:9200',
  agent: { agent: 'options' }
})

const client = new Client({
  node: 'http://localhost:9200',
  // the function takes as parameter the option
  // object passed to the Connection constructor
  agent: (opts) => new CustomAgent()
})

const client = new Client({
  node: 'http://localhost:9200',
  // Disable agent and keep-alive
  agent: false
})
```

<warning>
  If you have set [the `agent` option](#agent-config) on your client instance to a function and are using `UndiciConnection`the default [`Connection`](/docs/reference/elasticsearch/clients/javascript/advanced-config#_connection) value starting in 8.0all `caFingerprint` and `tls` options will be ignored. It is your responsibility to ensure that your custom agent will properly verify HTTPS connections.
</warning>


### `nodeFilter`

Type: `function`
Filter that indicates whether a node should be used for a request. Default function definition:
```js
function defaultNodeFilter (conn) {
  if (conn.roles != null) {
    if (
      // avoid master-only nodes
      conn.roles.master &&
      !conn.roles.data &&
      !conn.roles.ingest &&
      !conn.roles.ml
    ) return false
  }
  return true
}
```


### `nodeSelector`

Type: `function`
Default: `'round-robin'`
Custom selection strategy.
Options: `'round-robin'`, `'random'`, custom function
Custom function example:
```js
function nodeSelector (connections) {
  const index = calculateIndex()
  return connections[index]
}
```


### `generateRequestId`

Type: `function`
function to generate the request id for every request, it takes two parameters, the request parameters and options. By default, it generates an incremental integer for every request.
Custom function example:
```js
function generateRequestId (params, options) {
  // your id generation logic
  // must be syncronous
  return 'id'
}
```


### `name`

Type: `string, symbol`
Default: `elasticsearch-js`
The name to identify the client instance in the events.

### `opaqueIdPrefix`

Type: `string`
Default: `null`
A string that will be use to prefix any `X-Opaque-Id` header.
See [`X-Opaque-Id` support](/docs/reference/elasticsearch/clients/javascript/observability#_x_opaque_id_support) for more details.

### `headers`

Type: `object`
Default: `{}`
A set of custom headers to send in every request.

### `context`

Type: `object`
Default: `null`
A custom object that you can use for observability in your events. It will be merged with the API level context option.

### `enableMetaHeader`

Type: `boolean`
Default: `true`
If true, adds an header named `'x-elastic-client-meta'`, containing some minimal telemetry data, such as the client and platform version.

### `cloud`

Type: `object`
Default: `null`
Custom configuration for connecting to [Elastic Cloud](https://cloud.elastic.co). See [Authentication](https://www.elastic.co/docs/reference/elasticsearch/clients/javascript/connecting) for more details.
Cloud configuration example:
```js
const client = new Client({
  cloud: {
    id: '<cloud-id>'
  },
  auth: {
    username: 'elastic',
    password: 'changeme'
  }
})
```


### `disablePrototypePoisoningProtection`

Default: `true`
`boolean`, `'proto'`, `'constructor'` - The client can protect you against prototype poisoning attacks. For more information, refer to [Square Brackets are the Enemy](https://web.archive.org/web/20200319091159/https://hueniverse.com/square-brackets-are-the-enemy-ff5b9fd8a3e8?gi=184a27ee2a08). If needed, you can enable prototype poisoning protection entirely (`false`) or one of the two checks (`'proto'` or `'constructor'`). For performance reasons, it is disabled by default. To learn more, refer to the [`secure-json-parse` documentation](https://github.com/fastify/secure-json-parse).

### `caFingerprint`

Type: `string`
Default: `null`
If configured, verify that the fingerprint of the CA certificate that has signed the certificate of the server matches the supplied fingerprint. Only accepts SHA256 digest fingerprints.

### `maxResponseSize`

Type: `number`
Default: `null`
When configured, `maxResponseSize` verifies that the uncompressed response size is lower than the configured number. If it’s higher, the request will be canceled. The `maxResponseSize` cannot be higher than the value of `buffer.constants.MAX_STRING_LENGTH`.

### `maxCompressedResponseSize`

Type: `number`
Default: `null`
When configured, `maxCompressedResponseSize` verifies that the compressed response size is lower than the configured number. If it’s higher, the request will be canceled. The `maxCompressedResponseSize` cannot be higher than the value of `buffer.constants.MAX_STRING_LENGTH`.

### `redaction`

Type: `object`
Default: A configuration that will replace known sources of sensitive data in `Error` metadata
Options for how to redact potentially sensitive data from metadata attached to `Error` objects
<note>
  [Read about redaction](/docs/reference/elasticsearch/clients/javascript/advanced-config#redaction) for more details
</note>


### `serverMode`

Type: `string`
Default: `"stack"`
Setting to `"stack"` sets defaults assuming a traditional (non-serverless) Elasticsearch instance. Setting to `"serverless"` sets defaults to work more seamlessly with [Elastic Cloud Serverless](https://www.elastic.co/guide/en/serverless/current/intro.html), like enabling compression and disabling features that assume the possibility of multiple Elasticsearch nodes.