﻿---
title: Reading documents by id
description: Elasticsearch is all about search, but you may also want to access documents directly, knowing their identifier. The "get" request is meant for this. 
url: https://www.elastic.co/docs/reference/elasticsearch/clients/java/usage/reading
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch Java Client
---

# Reading documents by id
Elasticsearch is all about search, but you may also want to access documents directly, knowing their identifier. The "get" request is meant for this.
<note>
  See the [Elasticsearch API documentation](https://www.elastic.co/docs/api/doc/elasticsearch/operation/operation-get) for a full explanation of get requests.
</note>


## Reading a domain object

The example below reads the document with identifier `bk-1` from the `products` index.
The `get` request has two parameters:
- the first parameter is the actual request, built below with the fluent DSL
- the second parameter is the class we want the document’s JSON to be mapped to.

```java
GetResponse<Product> response = esClient.get(g -> g
    .index("products")
    .id("bk-1"),
    Product.class     
);

if (response.found()) {
    Product product = response.source();
    logger.info("Product name " + product.getName());
} else {
    logger.info ("Product not found");
}
```


## Reading raw JSON

When your index contains semi-structured data or if you don’t have a domain object definition, you can also read the document as raw JSON data.
Raw JSON data is just another class that you can use as the result type for the get request. In the example below we use Jackson’s `ObjectNode`. We could also have used any JSON representation that can be deserialized by the JSON mapper associated to the `ElasticsearchClient`.
```java
GetResponse<ObjectNode> response = esClient.get(g -> g
    .index("products")
    .id("bk-1"),
    ObjectNode.class    
);

if (response.found()) {
    ObjectNode json = response.source();
    String name = json.get("name").asText();
    logger.info("Product name " + name);
} else {
    logger.info("Product not found");
}
```

The source code for the examples above can be found in the [Java API Client tests](https://github.com/elastic/elasticsearch-java/tree/main/java-client/src/test/java/co/elastic/clients/documentation).