﻿---
title: Aggregations
description: An aggregation summarizes your data as metrics, statistics, or other analytics. In the example below we run an aggregation that creates a price histogram...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/java/usage/aggregations
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch Java Client
---

# Aggregations
An aggregation summarizes your data as metrics, statistics, or other analytics.
<note>
  See the [Elasticsearch documentation](https://www.elastic.co/docs/reference/aggregations) for a full explanation of aggregations.
</note>


## A simple aggregation

In the example below we run an aggregation that creates a price histogram from a product index, for the products whose name match a user-provided text. To achieve this, we use a search request that has a query (explained in [Searching for documents](https://www.elastic.co/docs/reference/elasticsearch/clients/java/usage/searching)) and an aggregation definition.
This example is an analytics-type aggregation where we do not want to use the matching documents. A general pattern for search requests used for analytics is to set the result `size` to zero and the target class for search results to `Void`.
If that same aggregation was used for to display products and the price histogram as drill-down facets, we would have set `size` to a non-zero value and used `Product` as the target class to process the results.
```java
String searchText = "bike";

Query query = MatchQuery.of(m -> m
    .field("name")
    .query(searchText)
)._toQuery();

SearchResponse<Void> response = esClient.search(b -> b
    .index("products")
    .size(0)
    .query(query)
    .aggregations("price-histogram", a -> a
        .histogram(h -> h
            .field("price")
            .interval(50.0)
        )
    ),
    Void.class
);
```

The response contains an aggregation result for each aggregation in the request.
```java
List<HistogramBucket> buckets = response.aggregations()
    .get("price-histogram")
    .histogram()
    .buckets().array();

for (HistogramBucket bucket: buckets) {
    logger.info("There are " + bucket.docCount() +
        " bikes under " + bucket.key());
}
```

The source code for the examples above can be found in the [Java API Client tests](https://github.com/elastic/elasticsearch-java/tree/main/java-client/src/test/java/co/elastic/clients/documentation).