﻿---
title: Getting started with the Elasticsearch Java client
description: This page guides you through the installation process of the Java client, shows you how to instantiate the client, and how to perform basic Elasticsearch...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/java/getting-started
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch Java Client
---

# Getting started with the Elasticsearch Java client
This page guides you through the installation process of the Java client, shows you how to instantiate the client, and how to perform basic Elasticsearch operations with it.

### Requirements

Java 17 or later.

### Installation


#### Installation in a Gradle project

```groovy
dependencies {
    implementation 'co.elastic.clients:elasticsearch-java:9.3.0'
}
```


#### Installation in a Maven project

In the `pom.xml` of your project, add the following repository definition and dependencies:
```xml
<project>
  <dependencies>

    <dependency>
      <groupId>co.elastic.clients</groupId>
      <artifactId>elasticsearch-java</artifactId>
      <version>9.3.0</version>
    </dependency>

  </dependencies>
</project>
```

Refer to the [Installation](https://www.elastic.co/docs/reference/elasticsearch/clients/java/setup/installation) page to learn more.

### Connecting

You can connect to the Elastic Cloud using an API key and the Elasticsearch endpoint.
```java
// URL and API key
String serverUrl = "https://localhost:9200";
String apiKey = "VnVhQ2ZHY0JDZGJrU...";

ElasticsearchClient esClient = ElasticsearchClient.of(b -> b
    .host(serverUrl)
    .apiKey(apiKey)
);

// Use the client...

// Close the client, also closing the underlying transport object and network connections.
esClient.close();
```

Your Elasticsearch endpoint can be found on the **My deployment** page of your deployment:
![Finding Elasticsearch endpoint](https://www.elastic.co/docs/reference/elasticsearch/clients/java/images/es-endpoint.jpg)
You can generate an API key on the **Management** page under Security.
![Create API key](https://www.elastic.co/docs/reference/elasticsearch/clients/java/images/create-api-key.png)
For other connection options, refer to the [Connecting](https://www.elastic.co/docs/reference/elasticsearch/clients/java/setup/connecting) section.

### Operations

Time to use Elasticsearch! This section walks you through the basic, and most important, operations of Elasticsearch. For more operations and more advanced examples, refer to the [*Using the Java API Client*](https://www.elastic.co/docs/reference/elasticsearch/clients/java/usage) page.

#### Creating an index

This is how you create the `product` index:
```java
esClient.indices().create(c -> c
    .index("products")
);
```


#### Indexing documents

This is a simple way of indexing a document, here a `Product` application object:
```java
Product product = new Product("bk-1", "City bike", 123.0);

IndexResponse response = esClient.index(i -> i
    .index("products")
    .id(product.getSku())
    .document(product)
);

logger.info("Indexed with version " + response.version());
```


#### Getting documents

You can get documents by using the following code:
```java
GetResponse<Product> response = esClient.get(g -> g
    .index("products")
    .id("bk-1"),
    Product.class     
);

if (response.found()) {
    Product product = response.source();
    logger.info("Product name " + product.getName());
} else {
    logger.info ("Product not found");
}
```


#### Searching documents

This is how you can create a single match query with the Java client:
```java
String searchText = "bike";

SearchResponse<Product> response = esClient.search(s -> s
        .index("products")
        .query(q -> q
            .match(t -> t
                .field("name")
                .query(searchText)
            )
        ),
    Product.class
);
```


#### Updating documents

This is how you can update a document, for example to add a new field:
```java
Product product = new Product("bk-1", "City bike", 123.0);

esClient.update(u -> u
        .index("products")
        .id("bk-1")
        .upsert(product),
    Product.class
);
```


#### Deleting documents

```java
esClient.delete(d -> d.index("products").id("bk-1"));
```


#### Deleting an index

```java
esClient.indices().delete(d -> d
    .index("products")
);
```


## Examples

The [examples](https://github.com/elastic/elasticsearch-java/tree/main/examples) folder in the Github repository contains full working examples showing how to set up and use the client.

## Further reading

- Learn more about the [*API conventions*](https://www.elastic.co/docs/reference/elasticsearch/clients/java/api-conventions) of the Java client.