﻿---
title: Creating API objects from JSON data
description: A common workflow during application development with Elasticsearch is to use the Kibana Developer Console to interactively prepare and test queries,...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/java/api-conventions/loading-json
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch Java Client
---

# Creating API objects from JSON data
A common workflow during application development with Elasticsearch is to use the Kibana Developer Console to interactively prepare and test queries, aggregations, index mappings and other complex API calls. This results in working JSON snippets that you may want to use in your application.
As translating these JSON snippets to Java code can be time-consuming and error-prone, most of the data classes in the Java API Client can be loaded from JSON text: object builders have `withJson()` methods that populate the builder from raw JSON. This also allows you to combine dynamically loaded JSON with programmatic construction of objects.
Under the hood, the `withJson()` methods call the object’s deserializer. The JSON text’s structure and value types therefore have to be correct for the target data structure. Using `withJson()` keeps the strong typing guarantees of the Java API Client.

## Examples


### Loading an index definition from a resource file

Consider a resource file `some-index.json` containing an index definition:
```json
{
  "mappings": {
    "properties": {
      "field1": { "type": "text" }
    }
  }
}
```

You can create an index from that definition as follows:
```java
InputStream input = this.getClass()
    .getResourceAsStream("some-index.json");

CreateIndexRequest req = CreateIndexRequest.of(b -> b
    .index("some-index")
    .withJson(input)
);

boolean created = esClient.indices().create(req).acknowledged();
```


### Ingesting documents from JSON files

Similarly, you can read documents to be stored in Elasticsearch from data files:
```java
FileReader file = new FileReader(new File(dataDir, "document-1.json"));

IndexRequest<JsonData> req;

req = IndexRequest.of(b -> b
    .index("some-index")
    .withJson(file)
);

esClient.index(req);
```


### Creating a search request combining JSON and programmatic construction

You can combine `withJson()` with regular calls to setter methods. The example below loads the query part of a search request from a `String` and programmatically adds an aggregation.
```java
Reader queryJson = new StringReader(
    "{" +
    "  \"query\": {" +
    "    \"range\": {" +
    "      \"@timestamp\": {" +
    "        \"gt\": \"now-1w\"" +
    "      }" +
    "    }" +
    "  }" +
    "}");

SearchRequest aggRequest = SearchRequest.of(b -> b
    .withJson(queryJson)
    .aggregations("max-cpu", a1 -> a1
        .dateHistogram(h -> h
            .field("@timestamp")
            .calendarInterval(CalendarInterval.Hour)
        )
        .aggregations("max", a2 -> a2
            .max(m -> m.field("host.cpu.usage"))
        )
    )
    .size(0)
);

Map<String, Aggregate> aggs = esClient
    .search(aggRequest, Void.class)
    .aggregations();
```


### Creating a search request from multiple JSON snippets

The `withJson()` methods are partial deserializers: the properties loaded from the JSON will set property values or replace the previous ones, but will not reset other properties not found in the JSON input. You can use this to combine multiple JSON snippets to build complex search requests. In the example below, we combine separate definitions of a query that selects some documents and an aggregation that is run on the results of this query.
```java
Reader queryJson = new StringReader(
    "{" +
    "  \"query\": {" +
    "    \"range\": {" +
    "      \"@timestamp\": {" +
    "        \"gt\": \"now-1w\"" +
    "      }" +
    "    }" +
    "  }," +
    "  \"size\": 100" +
    "}");

Reader aggregationJson = new StringReader(
    "{" +
    "  \"size\": 0, " +
    "  \"aggregations\": {" +
    "    \"hours\": {" +
    "      \"date_histogram\": {" +
    "        \"field\": \"@timestamp\"," +
    "        \"interval\": \"hour\"" +
    "      }," +
    "      \"aggregations\": {" +
    "        \"max-cpu\": {" +
    "          \"max\": {" +
    "            \"field\": \"host.cpu.usage\"" +
    "          }" +
    "        }" +
    "      }" +
    "    }" +
    "  }" +
    "}");

SearchRequest aggRequest = SearchRequest.of(b -> b
    .withJson(queryJson)
    .withJson(aggregationJson)
    .ignoreUnavailable(true)
);

Map<String, Aggregate> aggs = esClient
    .search(aggRequest, Void.class)
    .aggregations();
```

Notice that order matters when the JSON snippets have some common properties: just as when setting property values programmatically, the last value that is set for a property overwrites the previous one.
The source code for the examples above can be found in the [Java API Client tests](https://github.com/elastic/elasticsearch-java/tree/main/java-client/src/test/java/co/elastic/clients/documentation).