﻿---
title: Getting started
description: This page guides you through the installation process of the Go client, shows you how to instantiate the client, and how to perform basic Elasticsearch...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/go/getting-started
products:
  - Elasticsearch
  - Elasticsearch Client
  - Elasticsearch Go Client
---

# Getting started
This page guides you through the installation process of the Go client, shows you how to instantiate the client, and how to perform basic Elasticsearch operations with it. You can use the client with either a low-level API or a fully typed API. This getting started shows you examples of both APIs.

## Requirements

Go version 1.21+

## Installation

To install the latest version of the client, run the following command:
```shell
go get github.com/elastic/go-elasticsearch/v9@latest
```

Refer to the [_Installation_](https://www.elastic.co/docs/reference/elasticsearch/clients/go/installation) page to learn more.

## Connecting

<tab-set>
  <tab-item title="Low-level API">
    You can connect to the Elastic Cloud using an API key and the Elasticsearch endpoint for the low level API:
    ```go
    client, err := elasticsearch.NewClient(elasticsearch.Config{
        CloudID: "<CloudID>",
        APIKey: "<ApiKey>",
    })
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    You can connect to the Elastic Cloud using an API key and the Elasticsearch endpoint for the fully-typed API:
    ```go
    typedClient, err := elasticsearch.NewTypedClient(elasticsearch.Config{
        CloudID: "<CloudID>",
        APIKey:  "<ApiKey>",
    })
    ```
  </tab-item>
</tab-set>

Your Elasticsearch endpoint can be found on the **My deployment** page of your deployment:
![Finding Elasticsearch endpoint](https://www.elastic.co/docs/reference/elasticsearch/clients/go/images/es-endpoint.jpg)
You can generate an API key on the **Management** page under Security.
![Create API key](https://www.elastic.co/docs/reference/elasticsearch/clients/go/images/create-api-key.png)
For other connection options, refer to the [_Connecting_](https://www.elastic.co/docs/reference/elasticsearch/clients/go/connecting) section.

## Operations

Time to use Elasticsearch! This section walks you through the basic, and most important, operations of Elasticsearch. For more operations and more advanced examples, refer to the [Examples](https://www.elastic.co/docs/reference/elasticsearch/clients/go/examples) page.

### Creating an index

<tab-set>
  <tab-item title="Low-level API">
    This is how you create the `my_index` index with the low level API:
    ```go
    client.Indices.Create("my_index")
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    This is how you create the `my_index` index with the fully-typed API:
    ```go
    typedClient.Indices.Create("my_index").Do(context.TODO())
    ```
  </tab-item>
</tab-set>


### Indexing documents

<tab-set>
  <tab-item title="Low-level API">
    This is a simple way of indexing a document by using the low-level API:
    ```go
    document := struct {
        Name string `json:"name"`
    }{
        "go-elasticsearch",
    }
    data, _ := json.Marshal(document)
    client.Index("my_index", bytes.NewReader(data))
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    This is a simple way of indexing a document by using the fully-typed API:
    ```go
    document := struct {
        Name string `json:"name"`
    }{
        "go-elasticsearch",
    }
    typedClient.Index("my_index").
            Id("1").
            Request(document).
            Do(context.TODO())
    ```
  </tab-item>
</tab-set>


### Getting documents

<tab-set>
  <tab-item title="Low-level API">
    You can get documents by using the following code with the low-level API:
    ```go
    client.Get("my_index", "id")
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    This is how you can get documents by using the fully-typed API:
    ```go
    typedClient.Get("my_index", "id").Do(context.TODO())
    ```
  </tab-item>
</tab-set>


### Searching documents

<tab-set>
  <tab-item title="Low-level API">
    This is how you can create a single match query with the low-level API:
    ```go
    query := `{ "query": { "match_all": {} } }`
    client.Search(
        client.Search.WithIndex("my_index"),
        client.Search.WithBody(strings.NewReader(query)),
    )
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    This is how you can perform a single match query with the fully-typed API:
    ```go
    typedClient.Search().
        Index("my_index").
        Request(&search.Request{
            Query: &types.Query{MatchAll: &types.MatchAllQuery{}},
        }).
        Do(context.TODO())
    ```
  </tab-item>
</tab-set>


### Updating documents

<tab-set>
  <tab-item title="Low-level API">
    This is how you can update a document, for example to add a new field, by using the low-level API:
    ```go
    client.Update("my_index", "id", strings.NewReader(`{doc: { language: "Go" }}`))
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    This is how you can update a document with the fully-typed API:
    ```go
    typedClient.Update("my_index", "id").
        Request(&update.Request{
            Doc: json.RawMessage(`{ language: "Go" }`),
        }).Do(context.TODO())
    ```
  </tab-item>
</tab-set>


### Deleting documents

<tab-set>
  <tab-item title="Low-level API">
    ```go
    client.Delete("my_index", "id")
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    ```go
    typedClient.Delete("my_index", "id").Do(context.TODO())
    ```
  </tab-item>
</tab-set>


### Deleting an index

<tab-set>
  <tab-item title="Low-level API">
    ```go
    client.Indices.Delete([]string{"my_index"})
    ```
  </tab-item>

  <tab-item title="Fully-typed API">
    ```go
    typedClient.Indices.Delete("my_index").Do(context.TODO())
    ```
  </tab-item>
</tab-set>


## Further reading

- Learn more about the [_Typed API_](https://www.elastic.co/docs/reference/elasticsearch/clients/go/typed-api), a strongly typed Go API for Elasticsearch.