﻿---
title: Debug information
description: Every response from Elasticsearch.Net and NEST contains a DebugInformation property that provides a human readable description of what happened during...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/dotnet/troubleshoot/debug-information
products:
  - Elasticsearch
  - Elasticsearch .NET Client
  - Elasticsearch Client
---

# Debug information
Every response from Elasticsearch.Net and NEST contains a `DebugInformation` property that provides a human readable description of what happened during the request for both successful and failed requests
```csharp
var response = client.Search<Project>(s => s
    .Query(q => q
        .MatchAll()
    )
);

response.DebugInformation.Should().Contain("Valid NEST response");
```

This can be useful in tracking down numerous problems and can also be useful when filing an [issue](https://github.com/elastic/elasticsearch-net/issues) on the GitHub repository.

## Request and response bytes

By default, the request and response bytes are not available within the debug information, but can be enabled globally on Connection Settings by setting `DisableDirectStreaming`. This disables direct streaming of
1. the serialized request type to the request stream
2. the response stream to a deserialized response type

```csharp
var connectionPool = new SingleNodeConnectionPool(new Uri("http://localhost:9200"));

var settings = new ConnectionSettings(connectionPool)
    .DisableDirectStreaming(); 

var client = new ElasticClient(settings);
```

or on a *per request* basis
```csharp
var response = client.Search<Project>(s => s
    .RequestConfiguration(r => r
        .DisableDirectStreaming() 
    )
    .Query(q => q
        .MatchAll()
    )
);
```

Configuring `DisableDirectStreaming` on an individual request takes precedence over any global configuration.
There is typically a performance and allocation cost associated with disabling direct streaming since both the request and response bytes must be buffered in memory, to allow them to be exposed on the response call details.

## TCP statistics

It can often be useful to see the statistics for active TCP connections, particularly when trying to diagnose issues with the client. The client can collect the states of active TCP connections just before making a request, and expose these on the response and in the debug information.
Similarly to `DisableDirectStreaming`, TCP statistics can be collected for every request by configuring on `ConnectionSettings`
```csharp
var connectionPool = new SingleNodeConnectionPool(new Uri("http://localhost:9200"));

var settings = new ConnectionSettings(connectionPool)
    .EnableTcpStats(); 

var client = new ElasticClient(settings);
```

or on a *per request* basis
```csharp
var response = client.Search<Project>(s => s
    .RequestConfiguration(r => r
        .EnableTcpStats() 
    )
    .Query(q => q
        .MatchAll()
    )
);

var debugInformation = response.DebugInformation;
```

With `EnableTcpStats` set, the states of active TCP connections will now be included on the response and in the debug information.
The client includes a `TcpStats` class to help with retrieving more detail about active TCP connections should it be required
```csharp
var tcpStatistics = TcpStats.GetActiveTcpConnections(); 
var ipv4Stats = TcpStats.GetTcpStatistics(NetworkInterfaceComponent.IPv4); 
var ipv6Stats = TcpStats.GetTcpStatistics(NetworkInterfaceComponent.IPv6); 

var response = client.Search<Project>(s => s
    .Query(q => q
        .MatchAll()
    )
);
```

<note>
  Collecting TCP statistics may not be accessible in all environments, for example, Azure App Services. When this is the case, `TcpStats.GetActiveTcpConnections()` returns `null`.
</note>


## ThreadPool statistics

It can often be useful to see the statistics for thread pool threads, particularly when trying to diagnose issues with the client. The client can collect statistics for both worker threads and asynchronous I/O threads, and expose these on the response and in debug information.
Similar to collecting TCP statistics, ThreadPool statistics can be collected for all requests by configuring `EnableThreadPoolStats` on `ConnectionSettings`
```csharp
var connectionPool = new SingleNodeConnectionPool(new Uri("http://localhost:9200"));

var settings = new ConnectionSettings(connectionPool)
     .EnableThreadPoolStats(); 

var client = new ElasticClient(settings);
```

or on a *per request* basis
```csharp
var response = client.Search<Project>(s => s
     .RequestConfiguration(r => r
             .EnableThreadPoolStats() 
     )
     .Query(q => q
         .MatchAll()
     )
 );

var debugInformation = response.DebugInformation; 
```

With `EnableThreadPoolStats` set, the statistics of thread pool threads will now be included on the response and in the debug information.