﻿---
title: CRUD usage examples
description: This page helps you to understand how to perform various basic Elasticsearch CRUD (create, read, update, delete) operations using the .NET client. It...
url: https://www.elastic.co/docs/reference/elasticsearch/clients/dotnet/examples
products:
  - Elasticsearch
  - Elasticsearch .NET Client
  - Elasticsearch Client
---

# CRUD usage examples
This page helps you to understand how to perform various basic Elasticsearch CRUD (create, read, update, delete) operations using the .NET client. It demonstrates how to create a document by indexing an object into Elasticsearch, read a document back, retrieving it by ID or performing a search, update one of the fields in a document and delete a specific document.
These examples assume you have an instance of the `ElasticsearchClient` accessible via a local variable named `client` and several using directives in your C# file.
```csharp
using Elastic.Clients.Elasticsearch;

var client = new ElasticsearchClient(); 
```

The examples operate on data representing tweets. Tweets are modelled in the client application using a C# class named *Tweet* containing several properties that map to the document structure being stored in Elasticsearch.
```csharp
public class Tweet
{
    public int Id { get; set; } 
    public string User { get; set; }
    public DateTime PostDate { get; set; }
    public string Message { get; set; }
}
```


## Indexing a document

Documents can be indexed by creating an instance representing a tweet and indexing it via the client. In these examples, we will work with an index named *my-tweet-index*.
```csharp
var tweet = new Tweet 
{
    Id = 1,
    User = "stevejgordon",
    PostDate = new DateTime(2009, 11, 15),
    Message = "Trying out the client, so far so good?"
};

var response = await client.IndexAsync(tweet, x => x.Index("my-tweet-index")); 

if (response.IsValidResponse) 
{
    Console.WriteLine($"Index document with ID {response.Id} succeeded."); 
}
```


## Getting a document

```csharp
var response = await client.GetAsync<Tweet>(1, x => x.Index("my-tweet-index")); 

if (response.IsValidResponse)
{
    var tweet = response.Source; 
}
```


## Searching for documents

The client exposes a fluent interface and a powerful query DSL for searching.
```csharp
var response = await client.SearchAsync<Tweet>(s => s 
    .Indices("my-tweet-index") 
    .From(0)
    .Size(10)
    .Query(q => q
        .Term(t => t 
            .Field(x => x.User)
            .Value("stevejgordon")
        )
    )
);

if (response.IsValidResponse)
{
    var tweet = response.Documents.FirstOrDefault(); 
}
```

You may prefer using the object initializer syntax for requests if lambdas aren’t your thing.
```csharp
var request = new SearchRequest("my-tweet-index") 
{
    From = 0,
    Size = 10,
    Query = new Query { Term = new TermQuery { Field = "user", Value = "stevejgordon" } }
};

var response = await client.SearchAsync<Tweet>(request); 

if (response.IsValidResponse)
{
    var tweet = response.Documents.FirstOrDefault();
}
```


## Updating documents

Documents can be updated in several ways, including by providing a complete replacement for an existing document ID.
```csharp
tweet.Message = "This is a new message"; 

var response = await client.UpdateAsync<Tweet, Tweet>("my-tweet-index", 1, u => u
    .Doc(tweet)
); 

if (response.IsValidResponse)
{
    Console.WriteLine("Update document succeeded.");
}
```


## Deleting documents

Documents can be deleted by providing the ID of the document to remove.
```csharp
var response = await client.DeleteAsync("my-tweet-index", 1);

if (response.IsValidResponse)
{
    Console.WriteLine("Delete document succeeded.");
}
```