﻿---
title: ECS Logging Python installation
description: ecs-logging-python has formatters for the standard library logging module and the structlog package. You can exclude fields from being collected by using...
url: https://www.elastic.co/docs/reference/ecs/logging/python/installation
products:
  - ECS Logging
  - ECS Logging Python
---

# ECS Logging Python installation
```cmd
$ python -m pip install ecs-logging
```


## Getting started

`ecs-logging-python` has formatters for the standard library [`logging`](https://docs.python.org/3/library/logging.html) module and the [`structlog`](https://www.structlog.org/en/stable/) package.

### Standard library logging module

```python
import logging
import ecs_logging

# Get the Logger
logger = logging.getLogger("app")
logger.setLevel(logging.DEBUG)

# Add an ECS formatter to the Handler
handler = logging.StreamHandler()
handler.setFormatter(ecs_logging.StdlibFormatter())
logger.addHandler(handler)

# Emit a log!
logger.debug("Example message!", extra={"http.request.method": "get"})
```

```json
{
    "@timestamp": "2020-03-20T18:11:37.895Z",
    "log.level": "debug",
    "message": "Example message!",
    "ecs": {
        "version": "1.6.0"
    },
    "http": {
      "request": {
        "method": "get"
      }
    },
    "log": {
        "logger": "app",
        "origin": {
            "file": {
                "line": 14,
                "name": "test.py"
            },
            "function": "func"
        },
        "original": "Example message!"
    }
}
```


#### Excluding fields

You can exclude fields from being collected by using the `exclude_fields` option in the `StdlibFormatter` constructor:
```python
from ecs_logging import StdlibFormatter

formatter = StdlibFormatter(
    exclude_fields=[
        # You can specify individual fields to ignore:
        "log.original",
        # or you can also use prefixes to ignore
        # whole categories of fields:
        "process",
        "log.origin",
    ]
)
```


#### Limiting stack traces

The `StdlibLogger` automatically gathers `exc_info` into ECS `error.*` fields. If you’d like to control the number of stack frames that are included in `error.stack_trace` you can use the `stack_trace_limit` parameter (by default all frames are collected). Positive values include frames starting from the caller's frame. Negative values include the last `N` frames (closest to the error):
```python
from ecs_logging import StdlibFormatter

formatter = StdlibFormatter(
    # Only collects 3 stack frames
    stack_trace_limit=3,
)
formatter = StdlibFormatter(
    # Collects the last 2 frames (closest to the error)
    stack_trace_limit=-2,
)
formatter = StdlibFormatter(
    # Disable stack trace collection
    stack_trace_limit=0,
)
```


#### Controlling ASCII encoding

<applies-to>
  - : Generally available since 2.3
</applies-to>

By default, the `StdlibFormatter` escapes non-ASCII characters in the JSON output using Unicode escape sequences. If you want to preserve non-ASCII characters (such as Chinese, Japanese, emojis, etc.) in their original form, you can use the `ensure_ascii` parameter:
```python
from ecs_logging import StdlibFormatter

# Default behavior - non-ASCII characters are escaped
formatter = StdlibFormatter()
# Output: {"message":"Hello \\u4e16\\u754c"}

# Preserve non-ASCII characters
formatter = StdlibFormatter(ensure_ascii=False)
# Output: {"message":"Hello 世界"}
```

This is particularly useful when working with internationalized applications or when you need to maintain readability of logs containing non-ASCII characters.

### Structlog Example

Note that the structlog processor should be the last processor in the list, as it handles the conversion to JSON as well as the ECS field enrichment.
```python
import structlog
import ecs_logging

# Configure Structlog
structlog.configure(
    processors=[ecs_logging.StructlogFormatter()],
    wrapper_class=structlog.BoundLogger,
    context_class=dict,
    logger_factory=structlog.PrintLoggerFactory(),
)

# Get the Logger
logger = structlog.get_logger("app")

# Add additional context
logger = logger.bind(**{
    "http": {
        "version": "2",
        "request": {
            "method": "get",
            "bytes": 1337,
        },
    },
    "url": {
        "domain": "example.com",
        "path": "/",
        "port": 443,
        "scheme": "https",
        "registered_domain": "example.com",
        "top_level_domain": "com",
        "original": "https://example.com",
    }
})

# Emit a log!
logger.debug("Example message!")
```


#### Controlling ASCII encoding for Structlog

<applies-to>
  - : Generally available since 2.3
</applies-to>

Similar to `StdlibFormatter`, the `StructlogFormatter` also supports the `ensure_ascii` parameter to control whether non-ASCII characters are escaped:
```python
import structlog
import ecs_logging

# Configure Structlog with ensure_ascii=False to preserve non-ASCII characters
structlog.configure(
    processors=[ecs_logging.StructlogFormatter(ensure_ascii=False)],
    wrapper_class=structlog.BoundLogger,
    context_class=dict,
    logger_factory=structlog.PrintLoggerFactory(),
)

logger = structlog.get_logger("app")
logger.info("你好世界") 
```

```json
{
  "@timestamp": "2020-03-26T13:08:11.728Z",
  "ecs": {
    "version": "1.6.0"
  },
  "http": {
    "request": {
      "bytes": 1337,
      "method": "get"
    },
    "version": "2"
  },
  "log": {
    "level": "debug"
  },
  "message": "Example message!",
  "url": {
    "domain": "example.com",
    "original": "https://example.com",
    "path": "/",
    "port": 443,
    "registered_domain": "example.com",
    "scheme": "https",
    "top_level_domain": "com"
  }
}
```


## Elastic APM log correlation

`ecs-logging-python` supports automatically collecting  [ECS tracing fields](https://www.elastic.co/docs/reference/ecs/ecs-tracing) from the [Elastic APM Python agent](https://github.com/elastic/apm-agent-python) in order to [correlate logs to spans, transactions and traces](https://www.elastic.co/docs/reference/apm/agents/python/logs) in Elastic APM.
You can also quickly turn on ECS-formatted logs in your python app by setting [`LOG_ECS_REFORMATTING=override`](https://www.elastic.co/docs/reference/apm/agents/python/configuration#config-log_ecs_reformatting) in the Elastic APM Python agent.

## Install Filebeat

The best way to collect the logs once they are ECS-formatted is with [Filebeat](https://www.elastic.co/beats/filebeat):
<tab-set>
  <tab-item title="Log file">
    1. Follow the [Filebeat quick start](https://www.elastic.co/docs/reference/beats/filebeat/filebeat-installation-configuration)
    2. Add the following configuration to your `filebeat.yaml` file.
    For Filebeat 7.16+
    ```yaml
    filebeat.inputs:
    - type: filestream 
      paths: /path/to/logs.json
      parsers:
        - ndjson:
          overwrite_keys: true 
          add_error_key: true 
          expand_keys: true 

    processors: 
      - add_host_metadata: ~
      - add_cloud_metadata: ~
      - add_docker_metadata: ~
      - add_kubernetes_metadata: ~
    ```
    For Filebeat < 7.16
    ```yaml
    filebeat.inputs:
    - type: log
      paths: /path/to/logs.json
      json.keys_under_root: true
      json.overwrite_keys: true
      json.add_error_key: true
      json.expand_keys: true

    processors:
    - add_host_metadata: ~
    - add_cloud_metadata: ~
    - add_docker_metadata: ~
    - add_kubernetes_metadata: ~
    ```
  </tab-item>

  <tab-item title="Kubernetes">
    1. Make sure your application logs to stdout/stderr.
    2. Follow the [Run Filebeat on Kubernetes](https://www.elastic.co/docs/reference/beats/filebeat/running-on-kubernetes) guide.
    3. Enable [hints-based autodiscover](https://www.elastic.co/docs/reference/beats/filebeat/configuration-autodiscover-hints) (uncomment the corresponding section in `filebeat-kubernetes.yaml`).
    4. Add these annotations to your pods that log using ECS loggers. This will make sure the logs are parsed appropriately.

    ```yaml
    annotations:
      co.elastic.logs/json.overwrite_keys: true 
      co.elastic.logs/json.add_error_key: true 
      co.elastic.logs/json.expand_keys: true 
    ```
  </tab-item>

  <tab-item title="Docker">
    1. Make sure your application logs to stdout/stderr.
    2. Follow the [Run Filebeat on Docker](https://www.elastic.co/docs/reference/beats/filebeat/running-on-docker) guide.
    3. Enable [hints-based autodiscover](https://www.elastic.co/docs/reference/beats/filebeat/configuration-autodiscover-hints).
    4. Add these labels to your containers that log using ECS loggers. This will make sure the logs are parsed appropriately.

    ```yaml
    labels:
      co.elastic.logs/json.overwrite_keys: true 
      co.elastic.logs/json.add_error_key: true 
      co.elastic.logs/json.expand_keys: true 
    ```
  </tab-item>
</tab-set>

For more information, see the [Filebeat reference](https://www.elastic.co/docs/reference/beats/filebeat/configuring-howto-filebeat).