﻿---
title: Sanic Support
description: Incorporating Elastic APM into your Sanic project only requires a few easy steps. Install the Elastic APM agent using pip: or add elastic-apm to your...
url: https://www.elastic.co/docs/reference/apm/agents/python/sanic-support
products:
  - APM Agent
  - APM Python Agent
applies_to:
  - Serverless Observability projects: Generally available
  - Elastic Stack: Generally available
  - Application Performance Monitoring Agent for Python: Generally available
---

# Sanic Support
Incorporating Elastic APM into your Sanic project only requires a few easy steps.

## Installation

Install the Elastic APM agent using pip:
```bash
$ pip install elastic-apm
```

or add `elastic-apm` to your project’s `requirements.txt` file.

## Setup

To set up the agent, you need to initialize it with appropriate settings.
The settings are configured either via environment variables, or as initialization arguments.
You can find a list of all available settings in the [Configuration](https://www.elastic.co/docs/reference/apm/agents/python/configuration) page.
To initialize the agent for your application using environment variables:
```python
from sanic import Sanic
from elasticapm.contrib.sanic import ElasticAPM

app = Sanic(name="elastic-apm-sample")
apm = ElasticAPM(app=app)
```

To configure the agent using initialization arguments and Sanic’s Configuration infrastructure:
```python
# Create a file named external_config.py in your application
# If you want this module based configuration to be used for APM, prefix them with ELASTIC_APM_
ELASTIC_APM_SERVER_URL = "https://serverurl.example.com:443"
ELASTIC_APM_SECRET_TOKEN = "sometoken"
```

```python
from sanic import Sanic
from elasticapm.contrib.sanic import ElasticAPM

app = Sanic(name="elastic-apm-sample")
app.config.update_config("path/to/external_config.py")
apm = ElasticAPM(app=app)
```


## Usage

Once you have configured the agent, it will automatically track transactions and capture uncaught exceptions within sanic.
Capture an arbitrary exception by calling [`capture_exception`](/docs/reference/apm/agents/python/api-reference#client-api-capture-exception):
```python
from sanic import Sanic
from elasticapm.contrib.sanic import ElasticAPM

app = Sanic(name="elastic-apm-sample")
apm = ElasticAPM(app=app)

try:
    1 / 0
except ZeroDivisionError:
    apm.capture_exception()
```

Log a generic message with [`capture_message`](/docs/reference/apm/agents/python/api-reference#client-api-capture-message):
```python
from sanic import Sanic
from elasticapm.contrib.sanic import ElasticAPM

app = Sanic(name="elastic-apm-sample")
apm = ElasticAPM(app=app)

apm.capture_message('hello, world!')
```


## Performance metrics

If you’ve followed the instructions above, the agent has installed our instrumentation middleware which will process all requests through your app. This will measure response times, as well as detailed performance data for all supported technologies.
<note>
  Due to the fact that `asyncio` drivers are usually separate from their synchronous counterparts, specific instrumentation is needed for all drivers. The support for asynchronous drivers is currently quite limited.
</note>


### Ignoring specific routes

You can use the [`TRANSACTIONS_IGNORE_PATTERNS`](/docs/reference/apm/agents/python/configuration#config-transactions-ignore-patterns) configuration option to ignore specific routes. The list given should be a list of regular expressions which are matched against the transaction name:
```python
from sanic import Sanic
from elasticapm.contrib.sanic import ElasticAPM

app = Sanic(name="elastic-apm-sample")
apm = ElasticAPM(app=app, config={
    'TRANSACTIONS_IGNORE_PATTERNS': ['^GET /secret', '/extra_secret'],
})
```

This would ignore any requests using the `GET /secret` route and any requests containing `/extra_secret`.

## Extended Sanic APM Client Usage

Sanic’s contributed APM client also provides a few extendable way to configure selective behaviors to enhance the information collected as part of the transactions being tracked by the APM.
In order to enable this behavior, the APM Client middleware provides a few callback functions that you can leverage in order to simplify the process of generating additional contexts into the traces being collected.

| Callback Name             | Callback Invocation Format                                            | Expected Return Format                              | Is Async |
|---------------------------|-----------------------------------------------------------------------|-----------------------------------------------------|----------|
| transaction_name_callback | transaction_name_callback(request)                                    | string                                              | false    |
| user_context_callback     | user_context_callback(request)                                        | (username_string, user_email_string, userid_string) | true     |
| custom_context_callback   | custom_context_callback(request) or custom_context_callback(response) | dict(str=str)                                       | true     |
| label_info_callback       | label_info_callback()                                                 | dict(str=str)                                       | true     |


## Supported Sanic and Python versions

A list of supported [Sanic](/docs/reference/apm/agents/python/supported-technologies#supported-sanic) and [Python](/docs/reference/apm/agents/python/supported-technologies#supported-python) versions can be found on our [Supported Technologies](https://www.elastic.co/docs/reference/apm/agents/python/supported-technologies) page.
<note>
  Elastic APM only supports `asyncio` when using Python 3.7+
</note>