﻿---
title: Sum aggregation
description: A single-value metrics aggregation that sums up numeric values that are extracted from the aggregated documents. These values can be extracted either...
url: https://www.elastic.co/docs/reference/aggregations/search-aggregations-metrics-sum-aggregation
products:
  - Elasticsearch
---

# Sum aggregation
A `single-value` metrics aggregation that sums up numeric values that are extracted from the aggregated documents. These values can be extracted either from specific numeric or [histogram](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/histogram) fields.
Assuming the data consists of documents representing sales records we can sum the sale price of all hats with:
```json

{
  "query": {
    "constant_score": {
      "filter": {
        "match": { "type": "hat" }
      }
    }
  },
  "aggs": {
    "hat_prices": { "sum": { "field": "price" } }
  }
}
```

Resulting in:
```json
{
  ...
  "aggregations": {
    "hat_prices": {
      "value": 450.0
    }
  }
}
```

The name of the aggregation (`hat_prices` above) also serves as the key by which the aggregation result can be retrieved from the returned response.

## Script

If you need to get the `sum` for something more complex than a single field, run the aggregation on a [runtime field](https://www.elastic.co/docs/manage-data/data-store/mapping/runtime-fields).
```json

{
  "runtime_mappings": {
    "price.weighted": {
      "type": "double",
      "script": """
        double price = doc['price'].value;
        if (doc['promoted'].value) {
          price *= 0.8;
        }
        emit(price);
      """
    }
  },
  "query": {
    "constant_score": {
      "filter": {
        "match": { "type": "hat" }
      }
    }
  },
  "aggs": {
    "hat_prices": {
      "sum": {
        "field": "price.weighted"
      }
    }
  }
}
```


## Missing value

The `missing` parameter defines how documents that are missing a value should be treated. By default documents missing the value will be ignored but it is also possible to treat them as if they had a value. For example, this treats all hat sales without a price as being `100`.
```json

{
  "query": {
    "constant_score": {
      "filter": {
        "match": { "type": "hat" }
      }
    }
  },
  "aggs": {
    "hat_prices": {
      "sum": {
        "field": "price",
        "missing": 100
      }
    }
  }
}
```


## Histogram fields

When sum is computed on [histogram fields](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/histogram), the result of the aggregation is the sum of all elements in the `values` array multiplied by the number in the same position in the `counts` array.
For example, for the following index that stores pre-aggregated histograms with latency metrics for different networks:
```json

{
  "mappings": {
    "properties": {
      "latency_histo": { "type": "histogram" }
    }
  }
}


{
  "network.name" : "net-1",
  "latency_histo" : {
      "values" : [0.1, 0.2, 0.3, 0.4, 0.5],
      "counts" : [3, 7, 23, 12, 6]
   }
}


{
  "network.name" : "net-2",
  "latency_histo" : {
      "values" :  [0.1, 0.2, 0.3, 0.4, 0.5],
      "counts" : [8, 17, 8, 7, 6]
   }
}


{
  "aggs" : {
    "total_latency" : { "sum" : { "field" : "latency_histo" } }
  }
}
```

For each histogram field, the `sum` aggregation will add each number in the `values` array, multiplied by its associated count in the `counts` array.
Eventually, it will add all values for all histograms and return the following result:
```json
{
  "aggregations": {
    "total_latency": {
      "value": 28.8
    }
  }
}
```