﻿---
title: Nested aggregation
description: A special single bucket aggregation that enables aggregating nested documents. For example, lets say we have an index of products, and each product holds...
url: https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-nested-aggregation
products:
  - Elasticsearch
---

# Nested aggregation
A special single bucket aggregation that enables aggregating nested documents.
For example, lets say we have an index of products, and each product holds the list of resellers - each having its own price for the product. The mapping could look like:

```json

{
  "mappings": {
    "properties": {
      "resellers": { <1>
        "type": "nested",
        "properties": {
          "reseller": {
            "type": "keyword"
          },
          "price": {
            "type": "double"
          }
        }
      }
    }
  }
}
```

The following request adds a product with two resellers:
```json

{
  "name": "LED TV", <1>
  "resellers": [
    {
      "reseller": "companyA",
      "price": 350
    },
    {
      "reseller": "companyB",
      "price": 500
    }
  ]
}
```

The following request returns the minimum price a product can be purchased for:
```json

{
  "query": {
    "match": {
      "name": "led tv"
    }
  },
  "aggs": {
    "resellers": {
      "nested": {
        "path": "resellers"
      },
      "aggs": {
        "min_price": {
          "min": {
            "field": "resellers.price"
          }
        }
      }
    }
  }
}
```

As you can see above, the nested aggregation requires the `path` of the nested documents within the top level documents. Then one can define any type of aggregation over these nested documents.
Response:
```json
{
  ...
  "aggregations": {
    "resellers": {
      "doc_count": 2,
      "min_price": {
        "value": 350.0
      }
    }
  }
}
```

You can use a [`filter`](https://www.elastic.co/docs/reference/aggregations/search-aggregations-bucket-filter-aggregation) sub-aggregation to return results for a specific reseller.
```json

{
  "query": {
    "match": {
      "name": "led tv"
    }
  },
  "aggs": {
    "resellers": {
      "nested": {
        "path": "resellers"
      },
      "aggs": {
        "filter_reseller": {
          "filter": {
            "bool": {
              "filter": [
                {
                  "term": {
                    "resellers.reseller": "companyB"
                  }
                }
              ]
            }
          },
          "aggs": {
            "min_price": {
              "min": {
                "field": "resellers.price"
              }
            }
          }
        }
      }
    }
  }
}
```

The search returns:
```json
{
  ...
  "aggregations": {
    "resellers": {
      "doc_count": 2,
      "filter_reseller": {
        "doc_count": 1,
        "min_price": {
          "value": 500.0
        }
      }
    }
  }
}
```