﻿---
title: Typescript
description: Although this is not a requirement, we encourage if all new code is developed in Typescript. Kibana has crossed the 2m LoC mark. The current situation...
url: https://www.elastic.co/docs/extend/kibana/typescript
products:
  - Kibana
---

# Typescript
Although this is not a requirement, we encourage if all new code is developed in [Typescript](https://www.typescriptlang.org/).

## Project references

Kibana has crossed the 2m LoC mark. The current situation creates some scaling problems when the default out-of-the-box setup stops working. As a result, developers suffer from slow project compilation and IDE unresponsiveness. As a part of [Developer Experience project](https://github.com/elastic/kibana/projects/63), we are migrating our tooling to use built-in TypeScript features addressing the scaling problems - [project references](https://www.typescriptlang.org/docs/handbook/project-references.html) & [incremental builds](https://www.typescriptlang.org/docs/handbook/release-notes/typescript-3-4.html#faster-subsequent-builds-with-the---incremental-flag)
In a nutshell - instead of compiling the whole Kibana codebase at once, this setup enforces splitting the code base into independent projects that form a directed acyclic graph (DAG). This allows the TypeScript compiler (`tsc`) to apply several advanced optimizations:
- Every project emits `public` interfaces in the form of `d.ts` type declarations generated by the TypeScript compiler
- These generated `d.ts` type declarations are used whenever a referenced project is imported in a depending project
- This makes it possible to determine which project needs rebuilding when the source code has changed to use a more aggressive caching strategy.

More details are available in the [official docs](https://www.typescriptlang.org/docs/handbook/project-references.html)

### Caveats

This architecture imposes several limitations to which we must comply:
- Projects cannot have circular dependencies. Even though the Kibana platform doesn't support circular dependencies between Kibana plugins, TypeScript (and ES6 modules) allows it. So in theory, you can face a problem when migrating to the TS project references, leaving you with a circular dependency to fix. When building with TypeScript project references (using `node scripts/type_check` or `./node_modules/.bin/tsc -b`), circular dependencies between projects will be detected and reported, which must be resolved before migration can proceed.
- A project must emit its type declaration. It’s not always possible to generate a type declaration if the compiler cannot infer a type. There are two basic cases:
  1. Your plugin exports a type inferring an internal type declared in Kibana codebase. In this case, you’ll have to either export an internal type or to declare an exported type explicitly.
2. Your plugin exports something inferring a type from a 3rd party library that doesn’t export this type. To fix the problem, you have to declare the exported type manually.


### Prerequisites

Since project refs rely on generated `d.ts` files, the migration order does matter. You can migrate your plugin only when all the plugin dependencies already have migrated. It creates a situation where commonly used plugins (such as `data` or `kibana_react`) have to migrate first. Run `node scripts/find_plugins_without_ts_refs.js --id your_plugin_id` to get a list of plugins that should be switched to TS project refs to unblock your plugin migration.
Additionally, in order to migrate into project refs, you also need to make sure your plugin doesn't have circular dependencies with other plugins both on code and type imports. Circular dependencies between plugins are detected at runtime by the Kibana plugin system, which will prevent the application from starting if circular dependencies exist. When building with TypeScript project references using `node scripts/type_check` or `./node_modules/.bin/tsc -b`, circular dependencies between projects will also be caught during the build process and must be resolved.

### Implementation

- Make sure all the plugins listed as dependencies in **requiredPlugins**, **optionalPlugins** & **requiredBundles** properties of `kibana.json` manifest file have migrated to TS project references.
- Add `tsconfig.json` in the root folder of your plugin.

```
{
  "extends": "@kbn/tsconfig-base/tsconfig.json",
  "compilerOptions": {
    "outDir": "./target/types",
    "emitDeclarationOnly": true,
    "declaration": true,
    "declarationMap": true
  },
  "include": [
    // add all the folders containing files to be compiled
  ],
  "references": [
    { "path": "../../core/tsconfig.json" },
    // add references to other TypeScript projects your plugin depends on
  ]
}
```

If your plugin imports a file not listed in `include`, the build will fail with the next message `File ‘…’ is not listed within the file list of project …’. Projects must list all files or use an 'include' pattern.`
- Build your plugin using `./node_modules/.bin/tsc -b src/plugins/my_plugin` or `node scripts/type_check --project src/plugins/my_plugin/tsconfig.json`. Fix errors if the TypeScript compiler cannot generate type declarations for your project.
- Add your project reference to `references` property of `tsconfig.refs.json`
- Add your plugin to `references` property and plugin folder to `exclude` property of the `tsconfig.json` it used to belong to (for example, for `src/plugins/**` it’s `tsconfig.json`; for `x-pack/plugins/**` it’s `x-pack/tsconfig.json`).
- List the reference to your newly created project in all the Kibana `tsconfig.json` files that could import your project: `tsconfig.json`, `test/tsconfig.json`, `x-pack/tsconfig.json`, `x-pack/platform/test/tsconfig.json`. And in all the plugin-specific `tsconfig.refs.json` for dependent plugins.
- You can measure how your changes affect `tsc` compiler performance with `node --max-old-space-size=4096 ./node_modules/.bin/tsc -p tsconfig.json --extendedDiagnostics --noEmit`. Compare with `master` branch.

You can use [[https://github.com/elastic/kibana/pull/79446](https://github.com/elastic/kibana/pull/79446)](https://github.com/elastic/kibana/pull/79446) as an example.