﻿---
title: Localization for plugins outside the Kibana repo
description: To introduce localization for your plugin, use our i18n tool to create IDs and default messages. You can then extract these IDs with respective default...
url: https://www.elastic.co/docs/extend/kibana/external-plugin-localization
products:
  - Kibana
---

# Localization for plugins outside the Kibana repo
To introduce localization for your plugin, use our i18n tool to create IDs and default messages. You can then extract these IDs with respective default messages into localization JSON files for Kibana to use when running your plugin.

## Adding localization to your plugin

You must add a `translations` directory at the root of your plugin. This directory will contain the translation files that Kibana uses.
```shell
.
├── translations
│   ├── en.json
│   ├── ja-JP.json
│   ├── de-DE.json
│   ├── fr-FR.json
│   └── zh-CN.json
└── .i18nrc.json
```


## Using Kibana i18n tooling

To simplify the localization process, Kibana provides tools for the following functions:
- Verify all translations have translatable strings and extract default messages from templates
- Verify translation files and integrate them into Kibana

To use Kibana i18n tooling, create a `.i18nrc.json` file with the following configs:
- `paths`.  The directory from which the i18n translation IDs are extracted.
- `exclude`. The list of files to exclude while parsing paths.
- `translations`. The list of translations where JSON localizations are found.

```json
{
  "paths": {
    "myPlugin": "src/ui",
  },
  "exclude": [
  ],
  "translations": [
    "translations/zh-CN.json",
    "translations/fr-FR.json",
    "translations/de-DE.json",
    "translations/ja-JP.json"
  ]
}
```

An example Kibana `.i18nrc.json` is [here](https://github.com/elastic/kibana/blob/current/.i18nrc.json).
Full documentation about i18n tooling is [here](https://github.com/elastic/kibana/blob/current/src/dev/i18n/README.md).

## Extracting default messages

To extract the default messages from your plugin, run the following command:
```shell
node scripts/i18n_extract --output-dir ./translations --include-config ../kibana-extra/myPlugin/.i18nrc.json
```

This outputs a `en.json` file inside the `translations` directory. To localize other languages, clone the file and translate each string.

## Checking i18n messages

Checking i18n does the following:
- Checks all existing labels for violations.
- Takes translations from `.i18nrc.json` and compares them to the messages extracted and validated.
  - Checks for unused translations. If you remove a label that has a corresponding translation, you must also remove the label from the translations file.
- Checks for incompatible translations.  If you add or remove a new parameter from an existing string, you must also remove the label from the translations file.

To check your i18n translations, run the following command:
```shell
node scripts/i18n_check --fix --include-config ../kibana-extra/myPlugin/.i18nrc.json
```


## Implementing i18n in the UI

Kibana relies on ReactJS and requires localization in different environments (browser and NodeJS). The internationalization engine is framework agnostic and consumable in all parts of Kibana (ReactJS, and NodeJS).
To simplify internationalization in React, an additional abstraction is built around the I18n engine using [React-intl](https://github.com/yahoo/react-intl) for React.

### i18n for vanilla JavaScript

```js
import { i18n } from '@kbn/i18n';

export const HELLO_WORLD = i18n.translate('hello.wonderful.world', {
  defaultMessage: 'Greetings, planet Earth!',
});
```

Full details are [here](https://github.com/elastic/kibana/tree/main/src/platform/packages/shared/kbn-i18n#vanilla-js).

### i18n for React

To localize strings in React, use either `FormattedMessage` or `i18n.translate`.
```js
import { i18n } from '@kbn/i18n';
import { FormattedMessage } from '@kbn/i18n-react';

export const Component = () => {
  return (
    <div>
      {i18n.translate('xpack.someText', { defaultMessage: 'Some text' })}
      <FormattedMessage id="xpack.someOtherText" defaultMessage="Some other text">
      </FormattedMessage>
    </div>
  );
};
```

Full details are [here](https://github.com/elastic/kibana/tree/main/src/platform/packages/shared/kbn-i18n#react).

## Resources

To learn more about i18n tooling, see [i18n dev tooling](https://github.com/elastic/kibana/blob/master/src/dev/i18n/README.md).
To learn more about implementing i18n in the UI, use the following links:
- [i18n plugin](https://github.com/elastic/kibana/blob/master/src/platform/packages/shared/kbn-i18n/README.md)
- [i18n guidelines](https://github.com/elastic/kibana/blob/master/src/platform/packages/shared/kbn-i18n/GUIDELINE.md)