﻿---
title: Map custom regions with reverse geocoding
description: Maps comes with predefined regions that allow you to quickly visualize regions by metrics. Maps also offers the ability to map your own regions. You can...
url: https://www.elastic.co/docs/explore-analyze/visualize/maps/reverse-geocoding-tutorial
products:
  - Kibana
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Map custom regions with reverse geocoding
**Maps** comes with [predefined regions](https://maps.elastic.co/#file) that allow you to quickly visualize regions by metrics. **Maps** also offers the ability to map your own regions. You can use any region data you’d like, as long as your source data contains an identifier for the corresponding region.
But how can you map regions when your source data does not contain a region identifier? This is where reverse geocoding comes in. Reverse geocoding is the process of assigning a region identifier to a feature based on its location.
In this tutorial, you’ll use reverse geocoding to visualize United States Census Bureau Combined Statistical Area (CSA) regions by web traffic.
You’ll learn to:
- Upload custom regions.
- Reverse geocode with the Elasticsearch [enrich processor](https://www.elastic.co/docs/reference/enrich-processor/enrich-processor).
- Create a map and visualize CSA regions by web traffic.

When you complete this tutorial, you’ll have a map that looks like this:
![Map showing custom regions](https://www.elastic.co/docs/explore-analyze/images/kibana-csa_regions_by_web_traffic.png)


## Step 1: Index web traffic data

GeoIP is a common way of transforming an IP address to a longitude and latitude. GeoIP is roughly accurate on the city level globally and neighborhood level in selected countries. It’s not as good as an actual GPS location from your phone, but it’s much more precise than just a country, state, or province.
You’ll use the [web logs sample data set](/docs/explore-analyze#gs-get-data-into-kibana) that comes with Kibana for this tutorial. Web logs sample data set has longitude and latitude. If your web log data does not contain longitude and latitude, use [GeoIP processor](https://www.elastic.co/docs/reference/enrich-processor/geoip-processor) to transform an IP address into a [geo_point](https://www.elastic.co/docs/reference/elasticsearch/mapping-reference/geo-point) field.

## Step 2: Index Combined Statistical Area (CSA) regions

GeoIP level of detail is very useful for driving decision-making. For example, say you want to spin up a marketing campaign based on the locations of your users or show executive stakeholders which metro areas are experiencing an uptick of traffic.
That kind of scale in the United States is often captured with what the Census Bureau calls the Combined Statistical Area (CSA). CSA is roughly equivalent with how people intuitively think of which urban area they live in. It does not necessarily coincide with state or city boundaries.
CSAs generally share the same telecom providers and ad networks. New fast food franchises expand to a CSA rather than a particular city or municipality. Basically, people in the same CSA shop in the same IKEA.
To get the CSA boundary data:
1. Go to the [Census Bureau’s website](https://www.census.gov/geographies/mapping-files/time-series/geo/carto-boundary-file.html) and download the `cb_2018_us_csa_500k.zip` file.
2. Uncompress the zip file.
3. In Kibana, go to **Maps**.
4. Click **Create map**.
5. Click **Add layer**.
6. Click **Upload file**.
7. Use the file chooser to select the `.shp` file from the CSA shapefile folder.
8. Use the `.dbf` file chooser to select the `.dbf` file from the CSA shapefile folder.
9. Use the `.prj` file chooser to select the `.prj` file from the CSA shapefile folder.
10. Use the `.shx` file chooser to select the `.shx` file from the CSA shapefile folder.
11. Set index name to **csa** and click **Import file**.
12. When importing is complete, click **Add as document layer**.
13. Add Tooltip fields:
   1. Click **+ Add** to open the field select.
2. Select **NAME**, **GEOID**, and **AFFGEOID**.
3. Click **Add**.
14. Click **Keep changes**.

Looking at the map, you get a sense of what constitutes a metro area in the eyes of the Census Bureau.
![Map showing metro area](https://www.elastic.co/docs/explore-analyze/images/kibana-csa_regions.png)


## Step 3: Reverse geocoding

To visualize CSA regions by web log traffic, the web log traffic must contain a CSA region identifier. You’ll use Elasticsearch [enrich processor](https://www.elastic.co/docs/reference/enrich-processor/enrich-processor) to add CSA region identifiers to the web logs sample data set. You can skip this step if your source data already contains region identifiers.
1. Go to **Developer tools** using the navigation menu or the [global search field](https://www.elastic.co/docs/explore-analyze/find-and-organize/find-apps-and-objects).
2. In **Console**, create a [geo_match enrichment policy](https://www.elastic.co/docs/manage-data/ingest/transform-enrich/example-enrich-data-based-on-geolocation):
   ```js
   PUT /_enrich/policy/csa_lookup
   {
     "geo_match": {
       "indices": "csa",
       "match_field": "geometry",
       "enrich_fields": [ "GEOID", "NAME"]
     }
   }
   ```
3. To initialize the policy, run:
   ```js
   POST /_enrich/policy/csa_lookup/_execute
   ```
4. To create a ingest pipeline, run:
   ```js
   PUT _ingest/pipeline/lonlat-to-csa
   {
     "description": "Reverse geocode longitude-latitude to combined statistical area",
     "processors": [
       {
         "enrich": {
           "field": "geo.coordinates",
           "policy_name": "csa_lookup",
           "target_field": "csa",
           "ignore_missing": true,
           "ignore_failure": true,
           "description": "Lookup the csa identifier"
         }
       },
       {
         "remove": {
           "field": "csa.geometry",
           "ignore_missing": true,
           "ignore_failure": true,
           "description": "Remove the shape field"
         }
       }
     ]
   }
   ```
5. To update your existing data, run:
   ```js
   POST kibana_sample_data_logs/_update_by_query?pipeline=lonlat-to-csa
   ```
6. To run the pipeline on new documents at ingest, run:
   ```js
   PUT kibana_sample_data_logs/_settings
   {
     "index": {
       "default_pipeline": "lonlat-to-csa"
     }
   }
   ```
7. Go to **Discover**.
8. Set the data view to **Kibana Sample Data Logs**.
9. Open the [time filter](https://www.elastic.co/docs/explore-analyze/query-filter/filtering), and set the time range to the last 30 days.
10. Scan through the list of **Available fields** until you find the `csa.GEOID` field. You can also search for the field by name.
11. Click ![Add icon](https://www.elastic.co/docs/explore-analyze/images/kibana-add-icon.png) to toggle the field into the document table.
12. Find the *csa.NAME* field and add it to your document table.

Your web log data now contains `csa.GEOID` and `csa.NAME` fields from the matching **csa** region. Web log traffic not contained in a CSA region does not have values for `csa.GEOID` and `csa.NAME` fields.
![View of data in Discover](https://www.elastic.co/docs/explore-analyze/images/kibana-discover_enriched_web_log.png)


## Step 4: Visualize Combined Statistical Area (CSA) regions by web traffic

Now that our web traffic contains CSA region identifiers, you’ll visualize CSA regions by web traffic.
1. Go to **Maps**.
2. Click **Create map**.
3. Click **Add layer**.
4. Click **Choropleth**.
5. For **Boundaries source**:
   1. Select **Points, lines, and polygons from Elasticsearch**.
2. Set **Data view** to **csa**.
3. Set **Join field** to **GEOID**.
6. For **Statistics source**:
   1. Set **Data view** to **Kibana Sample Data Logs**.
2. Set **Join field** to **csa.GEOID.keyword**.
7. Click **Add and continue**.
8. Scroll to **Layer Style** and Set **Label** to **Fixed**.
9. Click **Keep changes**.
10. **Save** the map.
   1. Give the map a title.
2. Under **Add to dashboard**, select **None**.
3. Click **Save and add to library**.

![Final map showing custom regions](https://www.elastic.co/docs/explore-analyze/images/kibana-csa_regions_by_web_traffic.png)

Congratulations! You have completed the tutorial and have the recipe for visualizing custom regions. You can now try replicating this same analysis with your own data.