﻿---
title: Debug regex pattern matching failures in Painless
description: Regex operations in Painless can fail for several reasons: the regex feature is deactivated, there is incorrect matcher usage, or there are malformed...
url: https://www.elastic.co/docs/explore-analyze/scripting/painless-regex-pattern-matching-failures
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available
---

# Debug regex pattern matching failures in Painless
Regex operations in Painless can fail for several reasons: the regex feature is deactivated, there is incorrect matcher usage, or there are malformed regex patterns. This can occur when standard Java regex behavior is thought to apply directly to Painless.
Follow these guidelines to avoid [regex](https://www.elastic.co/docs/reference/scripting-languages/painless/painless-regexes) operation errors in your Painless scripts:
- **Call find() first:** Always use `matcher.find()` before accessing groups.
- **Enable regex:** Set `script.painless.regex.enabled=true` in the `elasticsearch.yml` settings file if regex is disabled.
- **Group numbering:** Use `group(0)` for the entire match, `group(1)` for first capture group, and so on.
- **Performance impact:** Regex operations can be expensive, especially with complex patterns.

For details, refer to the following sample error, solution, and the result when the solution is applied.

## Sample error

```json
{
  "error": {
    "root_cause": [
      {
        "type": "script_exception",
        "reason": "runtime error",
        "script_stack": [
          "java.base/java.util.regex.Matcher.checkMatch(Matcher.java:1850)",
          "java.base/java.util.regex.Matcher.group(Matcher.java:685)",
          """return /(?<=SHIPPER:).*?(?=\\n)/.matcher(orderLog).group(0);
    """,
          "                                                  ^---- HERE"
        ],
        "script": " ...",
        "lang": "painless",
        "position": {
          "offset": 176,
          "start": 126,
          "end": 191
        }
      }
    ],
    "type": "script_exception",
    "reason": "runtime error",
    "script_stack": [
      "java.base/java.util.regex.Matcher.checkMatch(Matcher.java:1850)",
      "java.base/java.util.regex.Matcher.group(Matcher.java:685)",
      """return /(?<=SHIPPER:).*?(?=\\n)/.matcher(orderLog).group(0);
    """,
      "                                                  ^---- HERE"
    ],
    "script": " ...",
    "lang": "painless",
    "position": {
      "offset": 176,
      "start": 126,
      "end": 191
    },
    "caused_by": {
      "type": "illegal_state_exception",
      "reason": "No match found"
    }
  },
  "status": 400
}
```


## Problematic code

```json
{
  "script": {
    "lang": "painless",
    "source": """
      String orderLog = "Order processing initiated\\nSHIPPER: SHIP-EX EXPRESS DELIVERY\\nTracking number generated";

      return /(?<=SHIPPER:).*?(?=\\n)/.matcher(orderLog).group(0);
    """
  }
}
```


## Root cause

The error occurs because the `group()` method is called on a Matcher before calling `find()`. You must explicitly search for matches before accessing groups. This is a common mistake when developers expect the matcher to automatically find matches.

## Solution: Call find() before accessing groups

Always call `find()` before using `group()` methods:
```json
POST _scripts/painless/_execute
{
  "script": {
    "lang": "painless",
    "source": """
      String orderLog = "Order processing initiated\\nSHIPPER: SHIP-EX EXPRESS DELIVERY\\nTracking number generated";

      Matcher m = /(?<=SHIPPER:).*?(?=\\n)/.matcher(orderLog);
      boolean found = m.find();

      return found ? m.group(0).trim() : "No match";
    """
  }
}
```


## Result

```json
{
  "result": "SHIP-EX EXPRESS DELIVERY"
}
```