﻿---
title: Slack action
description: Use the slack action to send messages to a Slack team’s channels or users. To send Slack messages, you need to configure at least one Slack account in...
url: https://www.elastic.co/docs/explore-analyze/alerts-cases/watcher/actions-slack
products:
  - Elasticsearch
applies_to:
  - Elastic Stack: Generally available
---

# Slack action
Use the `slack` action to send messages to a [Slack](https://slack.com/) team’s channels or users. To send Slack messages, you need to [configure at least one Slack account](#configuring-slack) in `elasticsearch.yml`.

## Configuring Slack actions

You configure Slack actions in the `actions` array. Action-specific attributes are specified using the `slack` keyword.
The following snippet shows a simple slack action definition:
```js
"actions" : {
  "notify-slack" : {
    "transform" : { ... },
    "throttle_period" : "5m",
    "slack" : {
      "message" : {
        "to" : [ "#admins", "@chief-admin" ], 
        "text" : "Encountered  {{ctx.payload.hits.total}} errors in the last 5 minutes (facepalm)" 
      }
    }
  }
}
```


## Using attachments to format Slack messages

In addition to sending simple text-based messages, you can use the Slack [attachment](https://api.slack.com/docs/attachments) mechanism to send formatted messages. Watcher leverages Slack attachments to enable you to dynamically populate templated messages from the execution context payload.
The following snippet shows a standard message attachment:
```js
"actions" : {
  "notify-slack" : {
    "throttle_period" : "5m",
    "slack" : {
      "account" : "team1",
      "message" : {
        "from" : "watcher",
        "to" : [ "#admins", "@chief-admin" ],
        "text" : "System X Monitoring",
        "attachments" : [
          {
            "title" : "Errors Found",
            "text" : "Encountered  {{ctx.payload.hits.total}} errors in the last 5 minutes (facepalm)",
            "color" : "danger"
          }
        ]
      }
    }
  }
}
```


## Dynamic attachments

To define an attachment template that is dynamically populated from the payload, you specify `dynamic_attachments` in the watch action. For example, a dynamic attachment could reference histogram buckets in the payload and build an attachment per bucket.
In the following example, the watch input executes a search with a date histogram aggregation and the Slack action:
1. Transforms the payload to a list where each item in the list holds the month, the user count for that month, and the color that represents the sentiment associated with that count (danger or bad).
2. Defines an attachment template that references items in the list generated by the transform.

```js
"input" : {
  "search" : {
    "request" : {
      "body" : {
        "aggs" : {
          "users_per_month" : {
            "date_histogram" : {
              "field" : "@timestamp",
              "interval" : "month"
            }
          }
        }
      }
    }
  }
},
...
"actions" : {
  "notify-slack" : {
    "throttle_period" : "5m",
    "transform" : {
      "script" : {
        "source" : "['items': ctx.payload.aggregations.users_per_month.buckets.collect(bucket -> ['count': bucket.doc_count, 'name': bucket.key_as_string, 'color': bucket.doc_count < 100 ? 'danger' : 'good'])]",
        "lang" : "painless"
      }
    },
    "slack" : {
      "account" : "team1",
      "message" : {
        "from" : "watcher",
        "to" : [ "#admins", "@chief-admin" ],
        "text" : "System X Monitoring",
        "dynamic_attachments" : {
          "list_path" : "ctx.payload.items" 
          "attachment_template" : {
            "title" : "{{month}}", 
            "text" : "Users Count: {{count}}",
            "color" : "{{color}}"
          }
        }
      }
    }
  }
}
```


## Slack action attributes


| Name                          | Required | Description                                                                                                                                                                                                                                                                                                               |
|-------------------------------|----------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|
| `message.from`                | no       | The sender name to display in the Slack message.                                    Overrides the incoming webhook’s configured name.                                                                                                                                                                                     |
| `message.to`                  | yes      | The channels and users you want to send the message                                    to. Channel names must start with `#` and user names                                    must start with `@`. Accepts a string value or an                                    array of string values.                               |
| `message.icon`                | no       | The icon to display in the Slack messages. Overrides                                    the incoming webhook’s configured icon. Accepts a                                    public URL to an image.                                                                                                                      |
| `message.text`                | yes      | The message content.                                                                                                                                                                                                                                                                                                      |
| `message.attachments`         | no       | Slack message attachments. Message attachments enable                                    you to create more richly-formatted messages. Specified                                    array as defined in the                                    [Slack attachments documentation](https://api.slack.com/docs/attachments). |
| `message.dynamic_attachments` | no       | Slack message attachments that can be populated                                    dynamically based on the current watch payload. For                                    more information, see                                    [Using attachments to format Slack messages](#slack-dynamic-attachment).               |
| `proxy.host`                  | no       | The proxy host to use (only in combination with `proxy.port`)                                                                                                                                                                                                                                                             |
| `proxy.port`                  | no       | The proxy port to use (only in combination with `proxy.host`)                                                                                                                                                                                                                                                             |


## Configuring Slack Accounts

You configure the accounts Watcher can use to communicate with Slack in the `xpack.notification.slack` namespace in [`elasticsearch.yml`](https://www.elastic.co/docs/deploy-manage/stack-settings).
You need a Slack App with the [Incoming Webhooks feature](https://api.slack.com/messaging/webhooks) to configure a Slack account. Use the generated webhook URL to set up your Slack account in Elasticsearch.
To configure a Slack account, at a minimum you need to specify the account name and webhook URL in the Elasticsearch keystore (see [secure settings](https://www.elastic.co/docs/deploy-manage/security/secure-settings)):
```shell
bin/elasticsearch-keystore add xpack.notification.slack.account.monitoring.secure_url
```

<warning>
  You can no longer configure Slack accounts using [`elasticsearch.yml`](https://www.elastic.co/docs/deploy-manage/stack-settings) settings. Use Elasticsearch's secure [keystore](https://www.elastic.co/docs/deploy-manage/security/secure-settings) method instead.
</warning>

You can specify defaults for the [Slack notification attributes](https://www.elastic.co/docs/reference/elasticsearch/configuration-reference/watcher-settings#slack-account-attributes):
```yaml
xpack.notification.slack:
  account:
    monitoring:
      message_defaults:
        from: x-pack
        to: notifications
        icon: http://example.com/images/watcher-icon.jpg
        attachment:
          fallback: "X-Pack Notification"
          color: "#36a64f"
          title: "X-Pack Notification"
          title_link: "https://www.elastic.co/guide/en/x-pack/current/index.html"
          text: "One of your watches generated this notification."
          mrkdwn_in: "pretext, text"
```

To notify multiple channels, create a webhook URL for each channel in Slack and multiple Slack accounts in Elasticsearch (one for each webhook URL).
If you configure multiple Slack accounts, you either need to configure a default account or specify which account the notification should be sent with in the `slack` action.
```yaml
xpack.notification.slack:
  default_account: team1
  account:
    team1:
      ...
    team2:
      ...
```