﻿---
title: ES|QL tools in Elastic Agent Builder
description: Create custom tools that execute parameterized ES|QL queries for precise data retrieval and analysis.
url: https://www.elastic.co/docs/explore-analyze/ai-features/agent-builder/tools/esql-tools
products:
  - Elastic Cloud Serverless
  - Elastic Observability
  - Elastic Security
  - Elasticsearch
  - Kibana
applies_to:
  - Elastic Cloud Serverless: Generally available
  - Elastic Stack: Generally available since 9.3, Preview in 9.2
---

# ES|QL tools in Elastic Agent Builder
ES|QL query tools enable you to create parameterized queries that execute directly against your Elasticsearch data. These custom tools provide precise control over data retrieval through templated [ES|QL](https://www.elastic.co/docs/reference/query-languages/esql) statements.

## When to use ES|QL tools

Use custom **ES|QL tools** when:
- You need precise control over the query logic
- Your use case involves repeatable analytical patterns
- You want to expose specific, parameterized queries to agents
- Results should be in a predictable tabular format
- You have well-defined data retrieval requirements

While agents can generate ES|QL queries dynamically using [index search tools](https://www.elastic.co/docs/explore-analyze/ai-features/agent-builder/tools/index-search-tools), custom ES|QL tools ensure syntax correctness and enforce critical business rules that an LLM might occasionally miss. For strategies to avoid data retrieval issues, refer to [Context length exceeded](https://www.elastic.co/docs/explore-analyze/ai-features/agent-builder/troubleshooting/context-length-exceeded).

## Key characteristics

- Execute pre-defined ES|QL queries with dynamic parameters
- Support typed parameters
- Return results in tabular format for structured data analysis
- Ideal for repeatable analytical queries with variable inputs


## Parameter types

ES|QL tools support the following parameter types:
- **String types**: `text`, `keyword`
- **Numeric types**: `long`, `integer`, `double`, `float`
- **Other types**: `boolean`, `date`, `object`, `nested`


## Parameter options

Parameters can be configured as:
- **Required**: Must be provided by the agent when calling the tool
- **Optional**: Can be omitted; uses `null` if no default is specified


## Query syntax

In your ES|QL query, reference parameters using the `?parameter_name` syntax. The agent will automatically interpolate parameter values when executing the query.

### Example

Here's an example ES|QL tool that searches for books using full-text search. `?search_terms` is a named parameter that the agent will provide when executing the query.
```esql
FROM books
| WHERE MATCH(title, ?search_terms)
| KEEP title, author, year
| LIMIT 10
```

You can ask the LLM to infer the parameters for the query or add them manually.
![Creating an ES|QL tool with a parameterized query](https://www.elastic.co/docs/explore-analyze/ai-features/agent-builder/images/create-esql-tool-query.png)

<dropdown title="Complex analytical query example">
  For high-stakes or complex analytical queries, pre-defining the ES|QL logic guarantees correctness and enforces business rules.**Tool name**: `Calculate Quarterly Revenue`**Description**: "Calculates confirmed revenue for a specific region broken down by quarter. Input requires a region code (e.g., 'US', 'EU')."**Query**:
  ```esql
  FROM finance-orders-*
  | WHERE order_status == "completed" AND region == ?region 
  | STATS total_revenue = SUM(amount) BY quarter
  | LIMIT 5
  ```
</dropdown>


## Best practices

- **Include [`LIMIT`](https://www.elastic.co/docs/reference/query-languages/esql/commands/limit) clauses**: Prevent returning excessive results by setting reasonable limits
- **Use meaningful parameter names**: Choose names that clearly indicate what the parameter represents (e.g., `start_date` instead of `date1`)
- **Define parameter types**: Ensure parameters have the correct type to avoid runtime errors
- **Provide clear descriptions**: Help agents understand when and how to use each parameter

For general guidance on naming tools and writing effective descriptions, refer to [Custom tools best practices](/docs/explore-analyze/ai-features/agent-builder/tools/custom-tools#best-practices).
<tip>
  If queries are slow or failing, you might be retrieving more data than the LLM can process. Refer to [Context length exceeded](https://www.elastic.co/docs/explore-analyze/ai-features/agent-builder/troubleshooting/context-length-exceeded) for tips on diagnosing and resolving these issues.
</tip>


## Limitations

ES|QL tools are subject to the current limitations of the ES|QL language itself. For more information, refer to [ES|QL tool limitations](/docs/explore-analyze/ai-features/agent-builder/limitations-known-issues#esql-limitations).

## ES|QL documentation

To learn more about the language, refer to the [ES|QL docs](https://www.elastic.co/docs/reference/query-languages/esql).