﻿---
title: Securing HTTP client applications
description: When connecting client applications to Elasticsearch, use these best practices: Always use HTTPS for all connections, Validate server certificates to...
url: https://www.elastic.co/docs/deploy-manage/security/httprest-clients-security
products:
  - Elasticsearch
applies_to:
  - Elastic Cloud Hosted: Generally available
  - Elastic Cloud on Kubernetes: Generally available
  - Elastic Cloud Enterprise: Generally available
  - Self-managed Elastic deployments: Generally available
---

# Securing HTTP client applications
When connecting client applications to Elasticsearch, use these best practices:
- Always use HTTPS for all connections
- Validate server certificates to prevent man-in-the-middle attacks
- Use API keys or token-based authentication rather than basic auth where possible
- Implement appropriate connection pooling and retry mechanisms
- Consider mutual TLS for high-security environments


## HTTP/REST clients and security

The Elasticsearch security features work with standard HTTP [basic authentication](https://en.wikipedia.org/wiki/Basic_access_authentication) headers to authenticate users. Since Elasticsearch is stateless, this header must be sent with every request:
```shell
Authorization: Basic <TOKEN> 
```

Alternatively, you can use [token-based authentication services](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/token-based-authentication-services).

### Client examples

This example uses `curl` without basic auth to create an index:
```shell
curl -XPUT 'localhost:9200/idx'
```

```js
{
  "error":  "AuthenticationException[Missing authentication token]",
  "status": 401
}
```

Since no user is associated with the request above, an authentication error is returned. Now we’ll use `curl` with basic auth to create an index as the `rdeniro` user:
```shell
curl --user rdeniro:taxidriver -XPUT 'localhost:9200/idx'
```

```js
{
  "acknowledged": true
}
```


### Secondary authorization

Some APIs support secondary authorization headers for situations where you want tasks to run with a different set of credentials. For example, you can send the following header in addition to the basic authentication header:
```shell
es-secondary-authorization: Basic <TOKEN> 
```

The `es-secondary-authorization` header has the same syntax as the `Authorization` header. It therefore also supports the use of [token-based authentication services](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/token-based-authentication-services). For example:
```shell
es-secondary-authorization: ApiKey <TOKEN> 
```


### Client libraries over HTTP

For more information about using security features with the language specific clients, refer to:
- [Java](https://www.elastic.co/docs/reference/elasticsearch/clients/java/setup/connecting)
- [JavaScript](https://www.elastic.co/docs/reference/elasticsearch/clients/javascript/connecting)
- [.NET](https://www.elastic.co/docs/reference/elasticsearch/clients/dotnet/configuration)
- [Perl](https://metacpan.org/pod/Search::Elasticsearch::Cxn::HTTPTiny#CONFIGURATION)
- [PHP](https://www.elastic.co/docs/reference/elasticsearch/clients/php/connecting)
- [Python](https://elasticsearch-py.readthedocs.io/en/master/#ssl-and-authentication)
- [Ruby](https://github.com/elasticsearch/elasticsearch-ruby/tree/master/elasticsearch-transport#authentication)