﻿---
title: Configuration for Fleet managed agents on Elastic Cloud on Kubernetes
description: Fleet-managed Elastic Agents must connect to Fleet Server to receive their configurations. You can deploy Fleet Server instances using ECKs Agent CRD...
url: https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/configuration-fleet
products:
  - Elastic Cloud on Kubernetes
applies_to:
  - Elastic Cloud on Kubernetes: Generally available
---

# Configuration for Fleet managed agents on Elastic Cloud on Kubernetes
Fleet-managed Elastic Agents must connect to Fleet Server to receive their configurations. You can deploy Fleet Server instances using ECKs Agent CRD with the appropriate configuration, as shown in [Fleet mode and Fleet Server](#k8s-elastic-agent-fleet-configuration-fleet-mode-and-fleet-server).
To know more about Fleet architecture and related components, check the Fleet [documentation](https://www.elastic.co/docs/reference/fleet/fleet-server).

## Fleet mode and Fleet Server

To run both Fleet Server and Elastic Agent in Fleet-managed mode, set the `mode` configuration element to `fleet`.
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: elastic-agent-sample
spec:
  mode: fleet
```

To run Fleet Server, set the `fleetServerEnabled` configuration element to `true`, as shown in this example:
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: fleet-server-sample
spec:
  mode: fleet
  fleetServerEnabled: true
```

You can leave the default value `false` for any other case.

## Configure Kibana

To have Fleet running properly, the following settings must be correctly set in the Kibana configuration:
```yaml
apiVersion: kibana.k8s.elastic.co/v1
kind: Kibana
metadata:
  name: kibana-sample
spec:
  config:
    xpack.fleet.agents.elasticsearch.hosts: ["<ELASTICSEARCH_HOST_URL>-es-http.default.svc:9200"]
    xpack.fleet.agents.fleet_server.hosts: ["<FLEET_SERVER_HOST_URL>-sample-agent-http.default.svc:8220"]
    xpack.fleet.packages:
      - name: system
        version: latest
      - name: elastic_agent
        version: latest
      - name: fleet_server
        version: latest
    xpack.fleet.agentPolicies:
      - name: Fleet Server on ECK policy
        id: eck-fleet-server
        namespace: default
        is_managed: true
        monitoring_enabled:
          - logs
          - metrics
        unenroll_timeout: 900
        package_policies:
        - name: fleet_server-1
          id: fleet_server-1
          package:
            name: fleet_server
      - name: Elastic Agent on ECK policy
        id: eck-agent
        namespace: default
        is_managed: true
        monitoring_enabled:
          - logs
          - metrics
        unenroll_timeout: 900
        is_default: true
        package_policies:
          - name: system-1
            id: system-1
            package:
              name: system
```

- `xpack.fleet.agents.elasticsearch.hosts` must point to the Elasticsearch cluster where Elastic Agents should send data. For ECK-managed Elasticsearch clusters ECK creates a Service accessible through `<ES_RESOURCE_NAME>-es-http.<ES_RESOURCE_NAMESPACE>.svc:9200` URL, where `ES_RESOURCE_NAME` is the name of Elasticsearch resource and `ES_RESOURCE_NAMESPACE` is the namespace it was deployed within. See [Storing local state in host path volume](/docs/deploy-manage/deploy/cloud-on-k8s/configuration-examples-standalone#k8s_storing_local_state_in_host_path_volume) for details on adjusting this field when running agent as non-root as it becomes required.
- `xpack.fleet.agents.fleet_server.hosts` must point to Fleet Server that Elastic Agents should connect to. For ECK-managed Fleet Server instances, ECK creates a Service accessible through `<FS_RESOURCE_NAME>-agent-http.FS_RESOURCE_NAMESPACE.svc:8220` URL, where `FS_RESOURCE_NAME` is the name of Elastic Agent resource with Fleet Server enabled and `FS_RESOURCE_NAMESPACE` is the namespace it was deployed in.
- `xpack.fleet.packages` are required packages to enable Fleet Server and Elastic Agents to enroll.
- `xpack.fleet.agentPolicies` policies are needed for Fleet Server and Elastic Agents to enroll to, check [https://www.elastic.co/guide/en/fleet/current/agent-policy.html](https://www.elastic.co/guide/en/fleet/current/agent-policy.html) for more information.


## Set referenced resources

Both Fleet Server and Elastic Agent in Fleet mode can be automatically set up with Fleet by ECK. The ECK operator can set up Fleet in Kibana (which otherwise requires manual steps) and enroll Fleet Server in the default Fleet Server policy. Elastic Agent can be automatically enrolled in the default Elastic Agent policy. To allow ECK to set this up, provide a reference to a ECK-managed Kibana through the `kibanaRef` configuration element.
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: fleet-server-sample
spec:
  kibanaRef:
    name: kibana
```

ECK can also facilitate the connection between Elastic Agents and a ECK-managed Fleet Server. To allow ECK to set this up, provide a reference to Fleet Server through the `fleetServerRef` configuration element.
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: elastic-agent-sample
spec:
  fleetServerRef:
    name: fleet-server-sample
```

Set the `elasticsearchRefs` element in your Fleet Server to point to the Elasticsearch cluster that will manage Fleet. Leave `elasticsearchRefs` empty or unset it for any Elastic Agent running in Fleet mode as the Elasticsearch cluster to target will come from Kibana's `xpack.fleet.agents.elasticsearch.hosts` configuration element.
<note>
  Currently, Elastic Agent in Fleet mode supports only a single output, so only a single Elasticsearch cluster can be referenced.
</note>

```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: fleet-server-sample
spec:
  elasticsearchRefs:
  - name: elasticsearch-sample
```

By default, every reference targets all instances in your Elasticsearch, Kibana and Fleet Server deployments, respectively. If you want to direct traffic to specific instances, refer to [*Traffic Splitting*](https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/requests-routing-to-elasticsearch-nodes) for more information and examples.

## Customize Elastic Agent configuration

In contrast to Elastic Agents in standalone mode, the configuration is managed through Fleet, and it cannot be defined through `config` or `configRef` elements with a few exceptions.
One of those exceptions is the configuration of providers as described in [advanced Agent configuration managed by Fleet](https://www.elastic.co/docs/reference/fleet/advanced-kubernetes-managed-by-fleet). When Elastic Agent is managed by Fleet and is orchestrated by ECK, the configuration of providers can simply be done through the `.spec.config` element in the Agent resource as of <applies-to>Elastic Stack: Generally available since 8.13</applies-to>:
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: elastic-agent
spec:
  config:
    fleet:
      enabled: true
    providers.kubernetes:
      add_resource_metadata:
        deployment: true
```


## Upgrade the Elastic Agent specification

You can upgrade the Elastic Agent version or change settings by editing the YAML specification file. ECK applies the changes by performing a rolling restart of the Agent’s Pods. Depending on the settings that you used, ECK will set up Fleet in Kibana, enrolls the agent in Fleet, or restarts Elastic Agent on certificate rollover.

## Choose the deployment model

Depending on the use case, Elastic Agent may need to be deployed as a [Deployment](https://kubernetes.io/docs/concepts/workloads/controllers/deployment/), a [DaemonSet](https://kubernetes.io/docs/concepts/workloads/controllers/daemonset/), or a [StatefulSet](https://kubernetes.io/docs/concepts/workloads/controllers/statefulSet/). To choose how to deploy your Elastic Agents, provide a `podTemplate` element under the `deployment` or the `daemonSet` element in the specification. If you choose the `deployment` option, you can additionally specify the [strategy](https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#strategy) used to replace old Pods with new ones.
Similarly, you can set the [update strategy](https://kubernetes.io/docs/tasks/manage-daemon/update-daemon-set/) when deploying as a DaemonSet. This allows you to control the rollout speed for new configuration by modifying the `maxUnavailable` setting:
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: elastic-agent-sample
spec:
  version: 9.3.0
  daemonSet:
    strategy:
      type: RollingUpdate
      rollingUpdate:
        maxUnavailable: 3
...
```

Refer to [Set compute resources for Beats and Elastic Agent](/docs/deploy-manage/deploy/cloud-on-k8s/manage-compute-resources#k8s-compute-resources-beats-agent) for more information on how to use the Pod template to adjust the resources given to Elastic Agent.

## Role Based Access Control for Elastic Agent

Some Elastic Agent features, such as the [Kubernetes integration](https://epr.elastic.co/package/kubernetes/0.2.8/), require that Agent Pods interact with Kubernetes APIs. This functionality requires specific permissions. Standard Kubernetes [RBAC](https://kubernetes.io/docs/reference/access-authn-authz/rbac/) rules apply. For example, to allow API interactions:
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: elastic-agent-sample
spec:
  version: 9.3.0
  elasticsearchRefs:
  - name: elasticsearch-sample
  daemonSet:
    podTemplate:
      spec:
        automountServiceAccountToken: true
        serviceAccountName: elastic-agent
...

apiVersion: v1
kind: ServiceAccount
metadata:
  name: elastic-agent
  namespace: default
---
apiVersion: rbac.authorization.k8s.io/v1
kind: ClusterRoleBinding
metadata:
  name: elastic-agent
subjects:
- kind: ServiceAccount
  name: elastic-agent
  namespace: default
roleRef:
  kind: ClusterRole
  name: elastic-agent
  apiGroup: rbac.authorization.k8s.io
```


## Deploy Elastic Agent in secured clusters

To deploy Elastic Agent in clusters with the Pod Security Policy admission controller enabled, or in [OpenShift](https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/k8s-openshift-agent) clusters, you might need to grant additional permissions to the Service Account used by the Elastic Agent Pods. Those Service Accounts must be bound to a Role or ClusterRole that has `use` permission for the required Pod Security Policy or Security Context Constraints. Different Elastic Agent Integrations might require different settings set in their PSP/[SCC](https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/k8s-openshift-agent).

## Customize Fleet Server Service

By default, ECK creates a Service for Fleet Server that Elastic Agents can connect through. You can customize it using the `http` configuration element. Check more information on how to [make changes](https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/accessing-services) to the Service and [customize](https://www.elastic.co/docs/deploy-manage/security/secure-cluster-communications) the TLS configuration.

## Control Fleet policy selection

ECK uses the default policy to enroll Elastic Agents in Fleet and the default Fleet Server policy to enroll Fleet Server. A different policy can be chosen by using the `policyID` attribute in the Elastic Agent resource:
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: fleet-server-sample
spec:
  policyID: my-custom-policy
...
```

Note that the environment variables related to policy selection mentioned in the Elastic Agent [docs](https://www.elastic.co/docs/reference/fleet/agent-environment-variables) like `FLEET_SERVER_POLICY_ID` will be managed by the ECK operator.

## Running as a non-root user

In order to run Elastic Agent as a non-root user you must choose how you want to persist data to the Agent’s volume.
1. Run Elastic Agent with an `emptyDir` volume. This has the downside of not persisting data between restarts of the Elastic Agent which can duplicate work done by the previous running Agent.
2. Run Elastic Agent with a `hostPath` volume in addition to a `DaemonSet` running as `root` that sets up permissions for the `agent` user.

In addition to these decisions, if you are running Elastic Agent in Fleet mode as a non-root user, you must configure `ssl.certificate_authorities` in each `xpack.fleet.outputs` to trust the CA of the Elasticsearch Cluster.
To run Elastic Agent with an `emptyDir` volume.
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: fleet-server
spec:
  deployment:
    podTemplate:
      spec:
        securityContext: 
          fsGroup: 1000
        volumes:
        - name: agent-data
          emptyDir: {}
...
```

To run Elastic Agent with a `hostPath` volume and a `DaemonSet` to maintain permissions.
```yaml
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: fleet-server-sample
  namespace: elastic-apps
spec:
  mode: fleet
  fleetServerEnabled: true
  deployment: {}
...
---
apiVersion: agent.k8s.elastic.co/v1alpha1
kind: Agent
metadata:
  name: elastic-agent-sample
  namespace: elastic-apps
spec:
  daemonSet: {}
...
---
apiVersion: apps/v1
kind: DaemonSet
metadata:
  name: manage-agent-hostpath-permissions
  namespace: elastic-apps
spec:
  selector:
    matchLabels:
      name: manage-agent-hostpath-permissions
  template:
    metadata:
      labels:
        name: manage-agent-hostpath-permissions
    spec:
     
      volumes:
        - hostPath:
            path: /var/lib/elastic-agent
            type: DirectoryOrCreate
          name: "agent-data"
      initContainers:
        - name: manage-agent-hostpath-permissions
         
          image: docker.io/bash:5.2.15
          resources:
            limits:
              cpu: 100m
              memory: 32Mi
          securityContext:
           
            runAsUser: 0
          volumeMounts:
            - mountPath: /var/lib/elastic-agent
              name: agent-data
          command:
          - 'bash'
          - '-e'
          - '-c'
          - |-
            # Adjust this with /var/lib/elastic-agent/YOUR-NAMESPACE/YOUR-AGENT-NAME/state
            # Multiple directories are supported for the fleet-server + agent use case.
            dirs=(
              "/var/lib/elastic-agent/default/elastic-agent/state"
              "/var/lib/elastic-agent/default/fleet-server/state"
              )
            for dir in ${dirs[@]}; do
              mkdir -p "${dir}"
              # chcon is only required when running an an SELinux-enabled/OpenShift environment.
              # chcon -Rt svirt_sandbox_file_t "${dir}"
              chmod g+rw "${dir}"
              # Gid 1000 is the default group at which the Agent container runs. Adjust as necessary if `runAsGroup` has been modified.
              chgrp 1000 "${dir}"
              if [ -n "$(ls -A ${dir} 2>/dev/null)" ]
              then
                # Gid 1000 is the default group at which the Agent container runs. Adjust as necessary if `runAsGroup` has been modified.
                chgrp 1000 "${dir}"/*
                chmod g+rw "${dir}"/*
              fi
            done
      containers:
        - name: sleep
          image: gcr.io/google-containers/pause-amd64:3.2
```

When running Agent in fleet mode as a non-root user Kibana must be configured in order to properly accept the CA of the Elasticsearch cluster.
```yaml
---
apiVersion: kibana.k8s.elastic.co/v1
kind: Kibana
metadata:
  name: kibana-sample
spec:
  config:
   
    xpack.fleet.agents.fleet_server.hosts: ["<FLEET_SERVER_HOST_URL>-agent-http.default.svc:8220"]
    xpack.fleet.outputs:
    - id: eck-fleet-agent-output-elasticsearch
      is_default: true
      name: eck-elasticsearch
      type: elasticsearch
      hosts:
      - "<ELASTICSEARCH-HOST_URL>-es-http.default.svc:9200" 
      ssl:
        certificate_authorities: ["/mnt/elastic-internal/elasticsearch-association/default/elasticsearch-sample/certs/ca.crt"] 
```