﻿---
title: Accessing services
description: To provide access to Elasticsearch, Kibana, and other Elastic Stack applications when applicable, ECK relies on Kubernetes services. All Elastic Stack...
url: https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/accessing-services
products:
  - Elastic Cloud on Kubernetes
applies_to:
  - Elastic Cloud on Kubernetes: Generally available
---

# Accessing services
To provide access to Elasticsearch, Kibana, and other Elastic Stack applications when applicable, ECK relies on [Kubernetes services](https://kubernetes.io/docs/concepts/services-networking/service/).
All Elastic Stack resources deployed by the ECK operator are secured by default. The operator sets up basic authentication and TLS to encrypt network traffic to, from, and within your Elasticsearch cluster.
This section explains how to access and customize the Kubernetes services and secrets created by ECK, covering topics such as:
- [Retrieving the `elastic` user password for basic authentication](#k8s-authentication)
- [Managing Kubernetes services](#k8s-kubernetes-service)
- [Obtaining the CA certificate and accessing the endpoint](#k8s-request-elasticsearch-endpoint)

For advanced use cases related to exposing and accessing orchestrated applications, see:
- [TLS encryption for cluster communications](https://www.elastic.co/docs/deploy-manage/security/secure-cluster-communications): Configuration options for the HTTP SSL certificates, including integration with certificate management systems such as [cert-manager](https://cert-manager.io/).
- [Service meshes](https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/service-meshes): Connect ECK and your managed deployments to service mesh implementations such as [Istio](https://istio.io) and [Linkerd](https://linkerd.io).
- [Requests routing to Elasticsearch nodes](https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/requests-routing-to-elasticsearch-nodes): Create custom services to expose different node types.
- [Use Ingress to expose Elasticsearch or Kibana](/docs/deploy-manage/deploy/cloud-on-k8s/managing-deployments-using-helm-chart#k8s-eck-stack-ingress): Helm based installation also facilitates the creation of Ingress resources.


## Retrieve the `elastic` user password

To access Elastic resources, the operator manages a default user named `elastic` with the `superuser` role. Its password is stored in a `Secret` named `<name>-elastic-user`.
Run the following command to obtain the password of the `elastic` user:
```sh
> kubectl get secret hulk-es-elastic-user -o go-template='{{.data.elastic | base64decode }}'
42xyz42citsale42xyz42
```

<note>
  Beware of copying this Secret as-is into a different namespace. Check [Common Problems: Owner References](/docs/troubleshoot/deployments/cloud-on-k8s/common-problems#k8s-common-problems-owner-refs) for more information.
</note>

For more information about handling built-in users on ECK deployments, refer to [Built-in users in self-managed clusters](https://www.elastic.co/docs/deploy-manage/users-roles/cluster-or-deployment-auth/built-in-users).

## Managing Kubernetes services

You can access Elastic resources by using native Kubernetes services that are not reachable from the public Internet by default.
For each resource, the operator manages a Kubernetes service named `<name>-[es|kb|apm|ent|agent]-http`, which is of type `ClusterIP` by default. `ClusterIP` exposes the service on a cluster-internal IP and makes the service only reachable within the cluster.
```sh
> kubectl get svc

NAME                TYPE           CLUSTER-IP      EXTERNAL-IP      PORT(S)          AGE
hulk-apm-http       ClusterIP      10.19.212.105   <none>           8200/TCP   1m
hulk-es-http        ClusterIP      10.19.252.160   <none>           9200/TCP   1m
hulk-kb-http        ClusterIP      10.19.247.151   <none>           5601/TCP   1m
```


### Allow public access

You can expose services in [different ways](https://kubernetes.io/docs/concepts/services-networking/service/#publishing-services-service-types) by specifying an `http.service.spec.type` in the `spec` of the resource manifest. On cloud providers which support external load balancers, you can set the `type` field to `LoadBalancer` to provision a load balancer for the `Service`, and populate the column `EXTERNAL-IP` after a short delay. Depending on the cloud provider, it may incur costs.
By default, the Elasticsearch service created by ECK is configured to route traffic to all Elasticsearch nodes in the cluster. Depending on your cluster configuration, you may want more control over the set of nodes that handle different types of traffic (query, ingest, and so on). Refer to [Requests routing to Elasticsearch nodes](https://www.elastic.co/docs/deploy-manage/deploy/cloud-on-k8s/requests-routing-to-elasticsearch-nodes) for more information.
Consider the following when customizing the Kubernetes services handled by ECK:
- When you change the `clusterIP` setting of the service, ECK deletes and re-creates the service, as `clusterIP` is an immutable field. Depending on your client implementation, this might result in a short disruption until the service DNS entries refresh to point to the new endpoints.
- If you change the service’s `port` to expose a different port externally, set `targetPort` to the container’s default listening port. Otherwise, Kubernetes uses the same value for both, resulting in failed connections. For example, default ports are `9200` for the Elasticsearch HTTP interface, and `5601` for Kibana.

```yaml
apiVersion: <kind>.k8s.elastic.co/v1
kind: <Kind>
metadata:
  name: hulk
spec:
  version: 9.3.0
  http:
    service:
      spec:
        type: LoadBalancer
```

```sh
> kubectl get svc

NAME                TYPE           CLUSTER-IP      EXTERNAL-IP      PORT(S)          AGE
hulk-apm-http       LoadBalancer   10.19.212.105   35.176.227.106   8200:31000/TCP   1m
hulk-es-http        LoadBalancer   10.19.252.160   35.198.131.115   9200:31320/TCP   1m
hulk-kb-http        LoadBalancer   10.19.247.151   35.242.197.228   5601:31380/TCP   1m
```


## Access the endpoint

You can access the Elasticsearch endpoint within or outside the Kubernetes cluster.
**Within the Kubernetes cluster**
1. Retrieve the CA certificate.
2. Retrieve the password of the `elastic` user.
3. Use the service name to access the endpoint.

```sh
NAME=hulk

kubectl get secret "$NAME-es-http-certs-public" -o go-template='{{index .data "tls.crt" | base64decode }}' > tls.crt
PW=$(kubectl get secret "$NAME-es-elastic-user" -o go-template='{{.data.elastic | base64decode }}')

curl --cacert tls.crt -u elastic:$PW https://$NAME-es-http:9200/
```

<tip>
  You can also use the examples in this section to access Kibana instead of Elasticsearch by adapting the secret and service names.
</tip>

**Outside the Kubernetes cluster**
1. Retrieve the CA certificate.
2. Retrieve the password of the `elastic` user.
3. Retrieve the IP of the `LoadBalancer` service.

```sh
NAME=hulk

kubectl get secret "$NAME-es-http-certs-public" -o go-template='{{index .data "tls.crt" | base64decode }}' > tls.crt
IP=$(kubectl get svc "$NAME-es-http" -o jsonpath='{.status.loadBalancer.ingress[].ip}')
PW=$(kubectl get secret "$NAME-es-elastic-user" -o go-template='{{.data.elastic | base64decode }}')

curl --cacert tls.crt -u elastic:$PW https://$IP:9200/
```